package tv.twitch.starshot64.recommendations.legacy

import android.content.Context
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.NetworkType
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkManager
import androidx.work.WorkerParameters
import com.apollographql.apollo.api.toInput
import com.apollographql.apollo.coroutines.await
import java.util.UUID
import java.util.concurrent.TimeUnit
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import timber.log.Timber
import tv.twitch.EmpStarshotLegacyAndroidRecommendationsQuery
import tv.twitch.fragment.StreamFragment
import tv.twitch.starshot64.net.graphql.newApolloClient
import tv.twitch.starshot64.recommendations.getRecommendationsStrategy
import tv.twitch.starshot64.util.getLanguageCode
import tv.twitch.starshot64.util.isSleepMode
import tv.twitch.starshot64.util.takeFirstNotNullN
import tv.twitch.type.RecommendationsContext

class NotificationRecommendationsWorker(context: Context, params: WorkerParameters) :
  CoroutineWorker(context, params) {

  companion object {
    private const val RECS_PER_ROW = 6
    private const val WORK_NAME = "NotificationRecsWork"
    private val logger = Timber.tag("NotifRecommWorker")

    fun scheduleImmediateWork(context: Context) {
      logger.d("scheduling immediate notifications recs work")

      val constraints = Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()

      val workRequest =
        OneTimeWorkRequestBuilder<NotificationRecommendationsWorker>().setConstraints(constraints)
          .build()

      WorkManager.getInstance(context).enqueue(workRequest)
    }

    fun schedulePeriodicWork(context: Context) {
      logger.d("scheduling periodic notifications recs work")

      val constraints = Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()
      val workRequest =
        PeriodicWorkRequestBuilder<NotificationRecommendationsWorker>(
          15,
          TimeUnit.MINUTES
        ).setConstraints(
          constraints
        ).build()

      WorkManager.getInstance(context).enqueueUniquePeriodicWork(
        WORK_NAME,
        ExistingPeriodicWorkPolicy.KEEP,
        workRequest
      )
    }
  }

  override suspend fun doWork(): Result = withContext(Dispatchers.IO) {
    logger.d("Starting work")

    if (isSleepMode(applicationContext)) {
      // Avoid unnecessary work and load on the backend if the device is asleep
      logger.d("Device is asleep, skipping recommendations update")
      Result.success()
    } else {
      val apollo = newApolloClient(applicationContext)

      val queryResult: kotlin.Result<EmpStarshotLegacyAndroidRecommendationsQuery.Data?> =
        try {
          val recsPerRow = 6
          val recsLocation = "TV_APPS"
          val recsPlatform = "android"
          val recsQuery = EmpStarshotLegacyAndroidRecommendationsQuery(
            recsPerRow.toInput(),
            getLanguageCode(),
            UUID.randomUUID().toString(),
            recsLocation,
            RecommendationsContext(platform = recsPlatform.toInput())
          )
          kotlin.Result.success(apollo.query(recsQuery).await().data)
        } catch (e: Exception) {
          kotlin.Result.failure(e)
        }

      if (queryResult.isSuccess && queryResult.getOrNull() != null) {
        logger.d("got some recs!")

        val result = queryResult.getOrNull()!!
        val recsStrategy = getRecommendationsStrategy()

        // Create stream recommendations
        val streamSources = ArrayList<Iterable<StreamFragment>?>()
        streamSources += result.currentUser?.followedLiveUsers?.edges?.mapNotNull { edge ->
          edge?.node?.stream?.fragments?.streamFragment
        }
        streamSources += result.recommendedStreams?.edges?.mapNotNull { edge ->
          edge.node?.fragments?.streamFragment
        }
        streamSources += result.featuredStreams?.mapNotNull { edge ->
          edge?.stream?.fragments?.streamFragment
        }

        val streamRecs = takeFirstNotNullN(RECS_PER_ROW, streamSources)
        streamRecs.forEach { stream ->
          logger.d("stream: ${stream.broadcaster!!.displayName}, viewers: ${stream.viewersCount}")
        }
        recsStrategy.updateRecommendedStreams(applicationContext, streamRecs)

        Result.success()
      } else {
        logger.e("failed to fetch recs")
        queryResult.exceptionOrNull()?.printStackTrace()
        Result.failure()
      }
    }
  }
}
