package tv.twitch.starshot64.recommendations.legacy

import android.annotation.TargetApi
import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import android.os.Build
import androidx.core.app.NotificationCompat
import androidx.recommendation.app.ContentRecommendation
import java.lang.reflect.Field
import timber.log.Timber
import tv.twitch.starshot64.BuildConfig
import tv.twitch.starshot64.R
import tv.twitch.starshot64.app.StarshotActivity

private const val CHANNEL_ID = "tv.twitch.android.app.recommendations"
private const val INTENT_REQUEST_CODE = 1
private const val FIRETV_WATCH_WITH_OPTION = 1

/**
 * Notifications on Fire OS 7+ require a notification channel to be created with which each notification
 * will be associated
 */
@TargetApi(Build.VERSION_CODES.O)
fun createNotificationChannel(context: Context) {
  try {
    val notificationManager =
      context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
    // Create the NotificationChannel, but only on API 26+ because
    // the NotificationChannel class is new and not in the support library
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
      val name = context.getString(R.string.recommended_streams)
      val description = context.getString(R.string.recommended_streams)
      val importance = NotificationManager.IMPORTANCE_DEFAULT
      val channel = NotificationChannel(CHANNEL_ID, name, importance)
      channel.description = description
      // Register the channel with the system; you can't change the importance
      // or other notification behaviors after this
      notificationManager.createNotificationChannel(channel)
    }
  } catch (ex: Exception) {
    Timber.e(ex, "Unable to create notification channel for recommendations")
  }
}

fun removeAllNotifications(context: Context) {
  val notificationManager =
    context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
  notificationManager.cancelAll()
}

private fun buildIntent(context: Context, uri: Uri): Intent {
  val streamIntent = Intent(context, StarshotActivity::class.java)
  streamIntent.data = uri
  return streamIntent
}

private fun buildPendingIntent(context: Context, intent: Intent, id: String): PendingIntent {
  intent.action = id
  return PendingIntent.getActivity(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT)
}

fun scheduleNotification(
  context: Context,
  id: String,
  title: String,
  description: String,
  previewBitmap: Bitmap,
  previewImageUriString: String,
  intentUri: Uri
) {
  val notification: Notification?
  val notificationIntent = buildIntent(context, intentUri)

  if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N_MR1) {
    // Use ContentRecommendation in Android >= 24.1
    val rec: ContentRecommendation = ContentRecommendation.Builder()
      .setIdTag(title)
      .setTitle(title)
      .setText(description)
      .setContentImage(previewBitmap)
      .setBackgroundImageUri(previewImageUriString)
      .setContentIntentData(
        ContentRecommendation.INTENT_TYPE_ACTIVITY, notificationIntent, INTENT_REQUEST_CODE, null
      )
      .setBadgeIcon(R.drawable.tv_glitch)
      .setStatus(ContentRecommendation.CONTENT_STATUS_READY)
      .build()

    notification = rec.getNotificationObject(context)

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && BuildConfig.FLAVOR == "amazon") {
      // Use reflection to add channel ID to our notification, as per Amazon docs
      try {
        val channel: Field = notification.javaClass.getDeclaredField("mChannelId")
        channel.isAccessible = true
        channel[notification] = CHANNEL_ID
      } catch (ex: Exception) {
        Timber.e(ex, "Error setting channel ID of notification")
      }
    }
  } else {
    // Use regular Notifications on older Androids
    notification = NotificationCompat.BigPictureStyle(
      NotificationCompat.Builder(context)
        .setContentTitle(title)
        .setContentText(description)
        .setPriority(NotificationCompat.PRIORITY_DEFAULT)
        .setLocalOnly(true)
        .setOngoing(true)
        .setColor(context.resources.getColor(R.color.twitch_purple))
        .setCategory(Notification.CATEGORY_RECOMMENDATION)
        .setLargeIcon(previewBitmap)
        .setSmallIcon(R.drawable.tv_glitch)
        .setContentIntent(buildPendingIntent(context, notificationIntent, id))
    ).build()

    if (notification == null) {
      Timber.e("Failed to create legacy style notification with ID: $id")
      return
    }
  }

  val nonNullNotification: Notification = notification!!

  if (BuildConfig.FLAVOR == "amazon") {
    // Amazon has some unique extra flags we need to add to our notification before we post it
    nonNullNotification.extras.putString(
      "com.amazon.extra.DISPLAY_NAME",
      context.resources.getString(R.string.app_name)
    )
    nonNullNotification.extras.putString("com.amazon.extra.LONG_DESCRIPTION", description)
    nonNullNotification.extras.putString("com.amazon.extra.PREVIEW_URL", previewImageUriString)
    // Tells FireTV to use "Watch with Twitch" as action
    nonNullNotification.extras.putIntegerArrayList(
      "com.amazon.extra.ACTION_OPTION",
      arrayListOf(
        FIRETV_WATCH_WITH_OPTION
      )
    )
    nonNullNotification.extras.putInt("com.amazon.extra.LIVE_CONTENT", 1)
  }

  val notificationManager =
    context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
  notificationManager.notify(title, id.toLong().toInt(), nonNullNotification)
}
