package tv.twitch.starshot64.util

import java.util.Timer
import java.util.TimerTask
import kotlin.math.pow

/**
 * A helper class to manage a delayed callback that may also require backoff for retries.
 */
class BackoffTimer(
  val name: String,
  private val maxBackoffMilliseconds: Long,
  private val maxJitterMilliseconds: Long,
  private val callback: () -> Unit
) {
  private val lock = Object()
  private var nextBackoffIndex = 0
  private var scheduledTask: TimerTask? = null

  /**
   * Whether or not the timer is in backoff mode.
   */
  fun isBackingOff(): Boolean {
    return nextBackoffIndex > 0
  }

  /**
   * Whether or not the timer is ticking.
   */
  fun isStarted(): Boolean {
    synchronized(lock) {
      return scheduledTask != null
    }
  }

  /**
   * Starts the timer without backoff.  This will clear backoff state.
   */
  fun start(delayMilliseconds: Long) {
    synchronized(lock) {
      nextBackoffIndex = 0
      schedule(delayMilliseconds)
    }
  }

  /**
   * Starts the timer in backoff mode.  The previous backoff state will be used to determine the
   * correct backoff for this timer.
   */
  fun startBackoff() {
    synchronized(lock) {
      nextBackoffIndex++

      var delay = 2.0.pow(nextBackoffIndex.toDouble())
      delay = delay.coerceAtMost(maxBackoffMilliseconds.toDouble())

      val jitteredDelay = applyJitter(delay.toLong())
      schedule(jitteredDelay)
    }
  }

  /**
   * Cancels the timer if ticking.  This does not clear backoff state.
   */
  fun cancel() {
    synchronized(lock) {
      if (scheduledTask != null) {
        scheduledTask!!.cancel()
        scheduledTask = null
      }
    }
  }

  private fun schedule(delayMilliseconds: Long) {
    scheduledTask = object : TimerTask() {
      override fun run() {
        var invoke = false
        synchronized(lock) {
          if (scheduledTask == this) {
            scheduledTask = null
            invoke = true
          }
        }

        if (invoke) {
          callback()
        }
      }
    }

    // Schedule the task
    Timer().schedule(scheduledTask, delayMilliseconds)
  }

  private fun applyJitter(x: Long): Long {
    return if (maxJitterMilliseconds == 0L) {
      x
    } else {
      x + LongRange(0, maxJitterMilliseconds).random()
    }
  }
}
