package tv.twitch.starshot64.util

import android.app.Activity
import android.content.Context
import android.content.SharedPreferences

private const val PREFS_FILE_USER = "user"
private const val PREFS_TWITCH_ACCESS_TOKEN = "twitch_access_token"
private const val PREFS_TWITCH_REFRESH_TOKEN = "twitch_refresh_token"

/**
 * Primitive used to synchronize access to credentials.  Recommendations may attempt to access
 * credentials from different threads at the same time they are being read/written in user
 * preferences.
 */
private val CREDENTIALS_LOCK = Object()

/**
 * Data store of Twitch credentials.
 */
data class TwitchCredentials(var accessToken: String = "", var refreshToken: String = "")

/**
 * The authentication data for the logged in user.  This will be null before the data is loaded.
 * If the user is not logged in then this will be non-null but the tokens will be "".
 */
private var gTwitchCredentials: TwitchCredentials? = null

/**
 * Retrieves the Twitch access token if available.
 */
fun getAccessToken(context: Context): String {
  val credentials = getCredentials(context)
  return credentials.accessToken
}

/**
 * Retrieves the bulk credentials data.
 */
fun getCredentials(context: Context): TwitchCredentials {
  synchronized(CREDENTIALS_LOCK) {
    var credentials: TwitchCredentials? = gTwitchCredentials

    // Not yet loaded so load now
    if (credentials == null) {
      credentials = loadCredentials(context)
    }

    // Never written before, setup the default values
    if (credentials == null) {
      credentials = TwitchCredentials()
      saveCredentials(context, credentials)
      gTwitchCredentials = credentials
    }

    return credentials.copy()
  }
}

/**
 * Sets the credentials in the global cache and saves them to user preferences.
 * This should be done when a user logs in via the web app.
 */
fun setCredentials(
  context: Context,
  data: TwitchCredentials
) {
  synchronized(CREDENTIALS_LOCK) {
    saveCredentials(context, data)
    setCachedCredentials(data)
  }
}

/**
 * Clears the credentials data and updates the global cache and user preference store.
 */
fun clearCredentials(context: Context) {
  synchronized(CREDENTIALS_LOCK) {
    val data = TwitchCredentials()
    saveCredentials(context, data)
    setCachedCredentials(data)
  }
}

/**
 * Saves the credentials to user preferences.
 */
private fun saveCredentials(context: Context, data: TwitchCredentials) {
  context.getSharedPreferences(PREFS_FILE_USER, Activity.MODE_PRIVATE)
    .edit()
    .putString(PREFS_TWITCH_ACCESS_TOKEN, data.accessToken)
    .putString(PREFS_TWITCH_REFRESH_TOKEN, data.refreshToken)
    .apply()
}

/**
 * Attempt to read the authentication data from user preferences.
 * @return The authentication data if stored, null otherwise.
 */
private fun loadCredentials(context: Context): TwitchCredentials? {
  val prefs: SharedPreferences = context.getSharedPreferences(
    PREFS_FILE_USER,
    Activity.MODE_PRIVATE
  )

  val accessToken = prefs.getString(PREFS_TWITCH_ACCESS_TOKEN, null)
  if (accessToken != null) {

    val result = TwitchCredentials()
    result.accessToken = accessToken

    val refreshToken = prefs.getString(PREFS_TWITCH_REFRESH_TOKEN, null)
    if (refreshToken != null) {
      result.refreshToken = refreshToken
    }

    return result
  } else {
    return null
  }
}

/**
 * Sets the globally cached credentials.
 * @param credentials The cached data.
 */
private fun setCachedCredentials(credentials: TwitchCredentials) {
  gTwitchCredentials = credentials
}
