package tv.twitch.starshot64.util

import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import java.util.TimeZone
import timber.log.Timber

/**
 * Takes a date from GraphQL and converts it to a Java Date
 *
 * Dates sent to us by Twitch APIs come in the 'yyyy-MM-ddTHH:mm:ssZ' format (ISO 8601)
 * Java's SimpleDateFormat only partially supports the ISO 8601 spec, and doesn't
 * recognize the standard format of appending 'Z' instead of a GMT offset to indicate
 * UTC time. If our date string contains the 'Z', convert it to a time zone format that
 * the parser knows how to handle.
 */
fun parseTwitchISO8601(dateString: String): Date {
  val dateFormat = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ", Locale.US)
  dateFormat.timeZone = TimeZone.getTimeZone("UTC")
  val correctedDateString = dateString.replace("Z$".toRegex(), "+0000")
  return try {
    dateFormat.parse(correctedDateString) ?: Date()
  } catch (pe: ParseException) {
    Timber.e("Failed to parse date string: $dateString, falling back to using current date")
    Date() // log an error and use current time if we somehow fail
  }
}

fun currentEpochInSeconds(): Long {
  val now = Date()
  return now.time / 1000
}
