package tv.twitch.starshot64.util

import android.content.Context
import android.os.PowerManager
import java.io.File
import java.io.RandomAccessFile
import java.util.UUID
import timber.log.Timber

/**
 * The globally cached device id.
 */
var gDeviceId: String? = null

/**
 * Obtains the device id.  This may be generated if first app launch or loaded from file.
 */
@Synchronized
fun getDeviceId(context: Context): String {

  var deviceId: String? = gDeviceId
  if (deviceId != null) {
    return deviceId
  }

  deviceId = loadDeviceId(context)

  // Cache globally
  gDeviceId = deviceId

  return deviceId
}

/**
 * Loads the device id from file.
 * This will automatically load the device ID from the previous TVAPPS app since the same storage
 * location is used.
 */
private fun loadDeviceId(context: Context): String {

  var deviceId: String? = null
  try {
    RandomAccessFile(getDeviceIdFile(context), "r").use { input ->
      val bytes = ByteArray(input.length().toInt())
      input.readFully(bytes)
      deviceId = String(bytes)
    }
  } catch (ex: Exception) {
    Timber.i(ex, "Failed to read device ID from storage")
  }

  // If still null, we failed to read an existing ID so just generate one
  if (deviceId == null) {
    deviceId = generateDeviceId()
    saveDeviceId(deviceId!!, context)
  }

  return deviceId!!
}

private fun getDeviceIdFile(context: Context): File {
  return File(context.filesDir, "unique_id")
}

/**
 * Saves the device id to file so it is available next time the app is launched.
 */
private fun saveDeviceId(deviceId: String, context: Context) {
  try {
    getDeviceIdFile(context).outputStream().use { output ->
      output.write(deviceId.toByteArray())
    }
  } catch (ex: Exception) {
    Timber.e(ex, "Failed to save device ID to storage")
  }
}

/**
 * Generate 40 random hex characters as per Twitch tracking spec.
 */
private fun generateDeviceId(): String {
  return generateRandomHexadecimal32Characters() +
    generateRandomHexadecimal32Characters().substring(0, 8)
}

private fun generateRandomHexadecimal32Characters(): String {
  // UUIDs are guaranteed to be 36 characters long, 4 of them being dashes
  return UUID.randomUUID().toString().replace("-", "")
}

/**
 * Determines whether or not the device is in sleep mode.
 */
fun isSleepMode(context: Context): Boolean {
  val powerManager = context.getSystemService(Context.POWER_SERVICE) as PowerManager
  return !powerManager.isInteractive
}
