package tv.twitch.starshot64.util

import kotlin.math.floor

/**
 * Formats the given Int into a readable count string reducing big numbers to 1.2k or 1.45M for example
 */
private const val ONE_BILLION_DOUBLE: Double = 1_000_000_000.0
private const val ONE_BILLION_LONG: Long = 1_000_000_000L
private const val POINT_ONE_BILLION: Int = 100_000_000
private const val ONE_BILLION_SHORT_FORMAT = "%.1fB"
private const val ONE_BILLION_POINT_ZERO_SHORT_FORMAT = "%.0fB"
private const val ONE_MILLION_DOUBLE: Double = 1_000_000.0
private const val ONE_MILLION_LONG: Long = 1_000_000L
private const val POINT_ONE_MILLION: Int = 100_000
private const val POINT_ZERO_ONE_MILLION: Int = 10_000
private const val ONE_MILLION_SHORT_FORMAT = "%.1fM"
private const val ONE_MILLION_POINT_ZERO_SHORT_FORMAT = "%.0fM"
private const val ONE_MILLION_LONG_FORMAT = "%.2f million"
private const val ONE_MILLION_POINT_ONE_LONG_FORMAT = "%.1f million"
private const val ONE_MILLION_POINT_ZERO_LONG_FORMAT = "%.0f million"
private const val ONE_THOUSAND_DOUBLE: Double = 1_000.0
private const val ONE_THOUSAND_LONG: Long = 1000L
private const val POINT_ONE_THOUSAND = 100
private const val ONE_THOUSAND_SHORT_FORMAT = "%.1fK"
private const val ONE_THOUSAND_POINT_ZERO_SHORT_FORMAT = "%.0fK"

fun unlocalizedFormat(
  count: Long,
  spellMillion: Boolean = false,
  useMax6Digits: Boolean = false
): String {
  return when {
    count >= ONE_BILLION_DOUBLE -> {
      val billions: Double = count.div(ONE_BILLION_DOUBLE)
      // checking using whole numbers avoids rounding errors
      if (count.rem(ONE_BILLION_LONG) >= POINT_ONE_BILLION) {
        ONE_BILLION_SHORT_FORMAT.format(billions)
      } else {
        ONE_BILLION_POINT_ZERO_SHORT_FORMAT.format(billions)
      }
    }
    count >= ONE_MILLION_DOUBLE -> {
      val millions: Double = count.div(ONE_MILLION_DOUBLE)
      val millionsRemainder: Long = count.rem(ONE_MILLION_LONG)
      if (spellMillion) {
        if (millionsRemainder >= POINT_ZERO_ONE_MILLION) {
          val pointOneMillionsRemainder = millionsRemainder.rem(POINT_ONE_MILLION)
          if (pointOneMillionsRemainder >= POINT_ZERO_ONE_MILLION) {
            ONE_MILLION_LONG_FORMAT.format(millions)
          } else {
            ONE_MILLION_POINT_ONE_LONG_FORMAT.format(millions)
          }
        } else {
          ONE_MILLION_POINT_ZERO_LONG_FORMAT.format(millions)
        }
      } else {
        if (millionsRemainder >= POINT_ONE_MILLION) {
          ONE_MILLION_SHORT_FORMAT.format(millions)
        } else {
          ONE_MILLION_POINT_ZERO_SHORT_FORMAT.format(millions)
        }
      }
    }
    count >= ONE_THOUSAND_DOUBLE && !useMax6Digits -> {
      val thousands: Double = count.div(ONE_THOUSAND_DOUBLE)
      val thousandsRemainder: Long = count.rem(ONE_THOUSAND_LONG)
      if (thousandsRemainder >= POINT_ONE_THOUSAND) {
        ONE_THOUSAND_SHORT_FORMAT.format(thousands)
      } else {
        ONE_THOUSAND_POINT_ZERO_SHORT_FORMAT.format(thousands)
      }
    }
    else -> count.toString()
  }
}

fun getSecondsFromMs(ms: Long): Long {
  // Can simplify to below after updating to Kotlin 1.5:
  // return ms.floorDiv(1000);
  return floor(ms.div(1000).toDouble()).toLong()
}
