package tv.twitch.starshot64.util

import android.webkit.CookieManager
import java.net.HttpCookie
import java.net.HttpURLConnection
import java.net.URL

/**
 * Takes a list of two strings and converts them to a cookie, where the first element is the name and second element is the value
 */
fun List<String>.toCookie(): HttpCookie? {
  return if (this.isNotEmpty() && this.size == 2) HttpCookie(this[0], this[1]) else null
}

/**
 * For a given URL, loads the stored cookies from the global cookie manager and parses them into a collection.
 *
 * As per the Android WebView documentation, expiration of cookies is handled automatically by the WebView, and
 * there's actually no good way to tell if cookies fetched from here would be expiring, since the returned value
 * from getCookie() is just a string of semicolon delimited pairs, where each pair is delimited by an equals sign.
 *
 * https://developer.android.com/reference/android/webkit/CookieManager#removeExpiredCookie()
 */
fun getCookies(url: String): List<HttpCookie> {
  val cookies = CookieManager.getInstance().getCookie(url)
  return if (cookies != null && cookies.isNotEmpty()) cookies.split(';').mapNotNull { pairStr ->
    pairStr.trim().split('=').toCookie()
  } else emptyList()
}

/**
 * For a given URL, sets the given cookie in the cookie manager.
 */
fun writeCookie(url: String, cookie: HttpCookie) {
  CookieManager.getInstance().setCookie(url, cookie.toString())
}

/**
 * Simple synchronous method for requesting an array of bytes from a URL
 */
fun simpleHttpGet(url: String): ByteArray? {
  val conn = URL(url).openConnection() as HttpURLConnection
  conn.connect()

  val inputStream = conn.inputStream

  if (inputStream != null) {
    return inputStream.readBytes()
  } else {
    return null
  }
}
