#!/usr/bin/python3
#
# This script aligns and signs an Android app bundle with the upload key.
# The upload keystore and certificate password are pulled from Sandstorm.
#
# To sign an AAB file:
#
#   sign_aab.py --input path_to_bundle.aab --output path_to_signed_bundle.aab
#

from sign_tools import *
from shutil import copyfile


def process(config):
    if config.input_path.endswith('apk'):
        raise Exception("Use sign_apk to sign APK files")

    dir = os.path.realpath(os.path.dirname(__file__))

    if not os.path.isfile(config.input_path):
        raise Exception("File not found: " + config.input_path)

    keystore_filename = 'upload.keystore'
    pem_filename = 'upload_keystore.pem'
    try:
        remove_if_exists(config.output_path)
        copyfile(config.input_path, config.output_path)

        # get upload key
        save_secret('platforms/Android/production/uploadkeystore',
                    dir, pem_filename)

        # get cert password
        keystore_pwd = keys.getSecret(
            'android', 'platforms/Android/production/uploadkeystorepassword')

        # generate p12 key
        alias = 'upload'
        generate_p12_key(pem_filename, keystore_pwd,
                         keystore_filename, alias, dir)

        # sign aab
        print("Signing aab")
        call_script_helper('', 'jarsigner', dir,
                           ['-keystore', 'file:' + keystore_filename,
                            '-storepass', keystore_pwd,
                            '-storetype', 'PKCS12',
                            config.output_path, alias
                            ])
    finally:
        # clean up
        remove_if_exists(os.path.join(dir, keystore_filename))
        remove_if_exists(os.path.join(dir, pem_filename))


# Process the command line arguments if run as the primary script
if __name__ == "__main__":

    # Parse the command line
    config = Configuration()
    parse_command_line(None, config)

    process(config)
