#!/usr/bin/python3
# This script aligns and signs an Android package with the prod cert key.
# The release keystore and certificate password are pulled from Sandstorm.
#
# To sign an APK file:
#
#   sign_apk.py --input path_to_package.apk --output path_to_package_signed.apk
#


from sign_tools import *


def process(config):
    if config.input_path.endswith('aab'):
        raise Exception("Use sign_aab to sign app bundle files")

    dir = os.getcwd()

    if not os.path.isfile(config.input_path):
        raise Exception("File not found: " + config.input_path,
                        ' cwd: ' + os.getcwd())

    keystore_filename = 'release.keystore'
    aligned_path = os.path.splitext(config.input_path)[0] + '-aligned.apk'
    try:
        remove_if_exists(config.output_path)
        remove_if_exists(aligned_path)
        remove_if_exists(keystore_filename)

        # get release api key
        pem_filename = 'releasekeystore.pem'
        save_secret('platforms/Android/production/releasekeystore',
                    dir, pem_filename)

        # get cert password
        keystore_pwd = keys.getSecret(
            'android', 'platforms/Android/production/certpassword')

        # align apk
        tools_dir = get_sdk_tools_dir()
        call_script_helper(tools_dir, 'zipalign', dir,
                           ['-v', '-f',
                            '4',
                            config.input_path,
                            aligned_path
                            ])

        # generate p12 key
        alias = 'twitch-release'
        generate_p12_key(pem_filename, keystore_pwd,
                         keystore_filename, alias, dir)

        # sign apk
        print("Signing apk")
        call_script_helper(tools_dir, 'apksigner', dir,
                           ['sign',
                            '--ks', keystore_filename,
                            '--ks-pass', 'pass:'+keystore_pwd,
                            '--out', config.output_path,
                            aligned_path
                            ])

    finally:
        # clean up
        remove_if_exists(os.path.join(dir, keystore_filename))
        remove_if_exists(os.path.join(dir, pem_filename))
        remove_if_exists(aligned_path)


# Process the command line arguments if run as the primary script
if __name__ == "__main__":

    # Parse the command line
    config = Configuration()
    parse_command_line(None, config)

    process(config)
