#!/usr/bin/python
#
# A collection of helper functions used by sign_apk.py and sign_aab.py
#

import argparse
import imp
import os
import shutil
import subprocess
import sys

# Load other scripts for use
script_dir = os.path.realpath(os.path.dirname(__file__))
keys = imp.load_source('keys', os.path.join(
    script_dir, '..', '..', 'scripts', 'keys.py'))

# Helper functions


def remove_if_exists(filename):
    try:
        os.remove(filename)
    except OSError:
        pass


# Import DEVNULL for muting verbose output
try:
    from subprocess import DEVNULL
except ImportError:
    # DEVNULL is not available in Python 2.x, so define it instead
    import os
    DEVNULL = open(os.devnull, 'wb')


def call_script_helper(dir, script_name, cwdir, args=[]):
    shell_args = [os.path.join(dir, script_name)] + args
    print(' '.join(shell_args) + ' cwd: ' + cwdir)
    proc = subprocess.Popen(shell_args, cwd=cwdir,
                            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    stdout, stderr = proc.communicate()

    stdout = stdout.decode('ascii')
    stderr = stderr.decode('ascii')

    if proc.returncode != 0:
        raise Exception(script_name + ' failed: ' +
                        str(proc.returncode) + ' out: ' + stdout + '\nerr: ' + stderr)


def write_to_file(data, dir, filename):
    with open(os.path.join(dir, filename), 'w') as file:
        file.write(data)


# Get paths for needed build tools, look for latest installed SDK
def get_sdk_tools_dir():
    sdk_home_name = 'ANDROID_HOME'
    if sdk_home_name not in os.environ:
        raise Exception('Cannot find ' + sdk_home_name +
                        ' environment variable')
    sdk_home = os.environ[sdk_home_name]
    tools_home = os.path.join(sdk_home, 'build-tools')
    return os.path.join(tools_home, next(os.walk(tools_home))[1][-1])


def save_secret(secret_path, dir, filename):
    keystore = keys.getSecret('android', secret_path)
    print('I got this secret: ' + keystore)
    if not keystore:
        raise Exception('Get keystore failed')
    remove_if_exists(os.path.join(dir, filename))
    write_to_file(keystore, dir, filename)


def generate_p12_key(pem_filename, keystore_pwd, keystore_filename, alias, dir):
    try:
        p12_filename = 'key.p12'
        remove_if_exists(os.path.join(dir, p12_filename))
        cat_process = subprocess.Popen(['cat', os.path.join(dir, pem_filename)],
                                       stdout=subprocess.PIPE)
        openssl_process = subprocess.Popen(['openssl', 'pkcs12',
                                            '-export',
                                            '-out', p12_filename,
                                            '-passin', 'pass:'+keystore_pwd,
                                            '-passout', 'pass:'+keystore_pwd,
                                            '-name', alias],
                                           stdin=cat_process.stdout)
        cat_process.stdout.close()
        openssl_process.wait()
        remove_if_exists(os.path.join(dir, keystore_filename))
        keytool_process = call_script_helper('', 'keytool', dir,
                                             ['-importkeystore',
                                              '-deststorepass', keystore_pwd,
                                              '-destkeystore', keystore_filename,
                                              '-deststoretype', 'PKCS12',
                                              '-srckeystore', p12_filename,
                                              '-srcstoretype', 'PKCS12',
                                              '-srcstorepass', keystore_pwd])
    finally:
        remove_if_exists(os.path.join(dir, p12_filename))


class Configuration:
    def __init__(self):
        self.input = None
        self.output = None


def parse_command_line(explicit_arguments, config):
    parser = argparse.ArgumentParser(description='Signs an Android APK file')

    parser.add_argument(
        '--input',
        required=True,
        metavar='input',
        help="Specifies path to the input apk."
    )

    parser.add_argument(
        '--output',
        required=True,
        metavar='output',
        help="Specifies path to the output apk."
    )

    args = parser.parse_args(explicit_arguments)

    if args.input:
        config.input_path = args.input

    if args.output:
        config.output_path = args.output
