#pragma once

#include <atomic>
#include <chrono>
#include <memory>
#include <vector>

/**
 * Returns the number of seconds since the epoch.
 * This needs to be defined in the application project so it is found at link time.
 */
uint64_t GetSecondsSinceEpoch();

std::chrono::milliseconds JitterTime(std::chrono::milliseconds baseMs, std::chrono::milliseconds width);

/**
 * Maintains a table of intervals used to manage the backoff that should be done
 * when retrying requests to a failing endpoint.
 * Usage:
 * - Create the table with the desired backoff.
 * - When the timed operation succeeds, call Reset() to reset the internal
 *   backoff. When the timed operation fails, call GetInterval() to start your
 *   timer and call Advance() so the next call to GetInterval() will use the next
 *   interval.
 */
class RetryBackoffTable {
 public:
  /**
   * Creates a default table with default jitter.
   */
  RetryBackoffTable();
  /**
   * Uses an explicit table and jitter.
   */
  RetryBackoffTable(
    const std::vector<std::chrono::milliseconds> &tableMilliseconds, std::chrono::milliseconds retryJitterWidthMs);
  /**
   * Computes an exponential backoff table and uses the given jitter.
   */
  RetryBackoffTable(std::chrono::milliseconds maxInterval, std::chrono::milliseconds retryJitterWidthMs);

  /**
   * Advance to the next retry interval.
   */
  void Advance();
  /**
   * Reset the retry attempt index.
   */
  void Reset();
  /**
   * Gets the next timer interval to use.
   */
  std::chrono::milliseconds GetInterval() const;

 private:
  void CreateTable(std::chrono::milliseconds maxInterval);

  std::vector<std::chrono::milliseconds> mBackOffTableMilliseconds;
  std::chrono::milliseconds mJitterMilliseconds;
  uint32_t mNextAttemptNumber;
};
