#include "core/httprequestutils.hpp"

#include "core/stringutilities.hpp"

#include <algorithm>
#include <cassert>
#include <regex>

namespace {
bool ShouldEncode(unsigned char c) {
  switch (c) {
    case '_':
    case '-':
    case '~':
    case '.':
      return false;
  }

  return isalnum(c) == 0;
}
}  // namespace

HttpParam::HttpParam(const std::string &name, const std::string &value) : paramName(name), paramValue(value) {}

HttpParam::HttpParam(const std::string &name, int value) : paramName(name) {
  char buffer[64];
  snprintf(buffer, sizeof(buffer), "%d", value);
  paramValue = buffer;
}

void UrlEncode(const std::string &inputString, std::stringstream &outputStream) {
  outputStream << std::hex << std::uppercase;
  for (auto currentCharacter : inputString) {
    if (ShouldEncode(static_cast<unsigned char>(currentCharacter))) {
      // Going from a negative char to an unsigned integer will cause the
      // additional 3 bytes to be padded full of 1s. Attempting to use an
      // unsigned char directly will cause the stringstream << operator to
      // insert the ASCII symbol the char maps to.
      outputStream << "%" << static_cast<unsigned int>(static_cast<unsigned char>(currentCharacter));
    } else {
      outputStream << currentCharacter;
    }
  }
}

std::string UrlEncode(const std::string &inputString) {
  std::stringstream stream;
  UrlEncode(inputString, stream);
  return stream.str();
}

void UrlDecode(const std::string &inputString, std::stringstream &outputStream) {
  outputStream << std::hex << std::uppercase;
  size_t index = 0;
  while (index < inputString.size()) {
    if (inputString[index] == '%') {
      // Convert the next 2 hex digits into a character
      if (inputString.size() > index + 2) {
        char hexValue[4];
        hexValue[0] = inputString.data()[index + 1];
        hexValue[1] = inputString.data()[index + 2];
        hexValue[2] = '\0';

        int intValue = 0;
        sscanf(hexValue, "%x", &intValue);

        outputStream << static_cast<char>(intValue);

        index += 3;
      }
      // Invalid encoding
      else {
        return;
      }
    } else {
      outputStream << inputString[index];
      index++;
    }
  }
}

void UrlDecode(const std::string &input, std::string &result) {
  std::stringstream sstream;
  UrlDecode(input, sstream);
  result = sstream.str();
}

std::string UrlDecode(const std::string &inputString) {
  std::stringstream sstream;
  UrlDecode(inputString, sstream);
  return sstream.str();
}

std::string BuildUrlEncodedRequestParams(const std::vector<std::pair<std::string, std::string>> &requestParams) {
  std::stringstream paramsStream;
  for (auto it = requestParams.begin(); it != requestParams.end(); ++it) {
    if (it != requestParams.begin()) {
      paramsStream << "&";
    }
    UrlEncode(it->first, paramsStream);
    paramsStream << "=";
    UrlEncode(it->second, paramsStream);
  }
  return paramsStream.str();
}

std::string BuildUrlEncodedRequestParams(const std::vector<HttpParam> &requestParams) {
  std::stringstream paramsStream;
  for (auto it = requestParams.begin(); it != requestParams.end(); ++it) {
    if (it != requestParams.begin()) {
      paramsStream << "&";
    }
    UrlEncode(it->paramName, paramsStream);
    paramsStream << "=";
    UrlEncode(it->paramValue, paramsStream);
  }
  return paramsStream.str();
}

std::string BuildUrlEncodedRequestParams(const std::map<std::string, std::string> &requestParams) {
  std::stringstream paramsStream;
  for (auto it = requestParams.begin(); it != requestParams.end(); ++it) {
    if (it != requestParams.begin()) {
      paramsStream << "&";
    }
    UrlEncode(it->first, paramsStream);
    paramsStream << "=";
    UrlEncode(it->second, paramsStream);
  }
  return paramsStream.str();
}

std::string BuildHttpHeader(const std::vector<HttpParam> &headerParams) {
  std::stringstream headerStream;

  for (auto it = headerParams.begin(); it != headerParams.end(); ++it) {
    headerStream << it->paramName.c_str();
    headerStream << ":";
    headerStream << it->paramValue.c_str();
    headerStream << "\r\n";  // Header parameters are seperated by CRLF
  }

  return headerStream.str();
}

bool ContainsHttpParameter(const std::vector<HttpParam> &headers, const std::string &name) {
  std::string lowerName = name;
  (void)std::transform(lowerName.begin(), lowerName.end(), lowerName.begin(), ::tolower);

  auto iter = std::find_if(headers.begin(), headers.end(), [&lowerName](const HttpParam &p) -> bool {
    std::string lowerParam = p.paramName;
    (void)std::transform(lowerParam.begin(), lowerParam.end(), lowerParam.begin(), ::tolower);

    return lowerParam == lowerName;
  });

  return iter != headers.end();
}

void SplitHttpParameters(const std::string &parameterString, std::vector<std::pair<std::string, std::string>> &result) {
  size_t currentIndex = 0;

  while (true) {
    auto equalsIndex = parameterString.find('=', currentIndex);

    if (equalsIndex == std::string::npos) {
      break;
    } else {
      auto valueIndex = equalsIndex + 1;
      std::string key;
      UrlDecode(parameterString.substr(currentIndex, equalsIndex - currentIndex), key);

      std::string value;
      auto ampersandIndex = parameterString.find('&', valueIndex);

      // If we didn't find the ampersand, we just get a substring to the end of
      // the string.
      auto valueLength = (ampersandIndex == std::string::npos) ? std::string::npos : ampersandIndex - valueIndex;
      UrlDecode(parameterString.substr(equalsIndex + 1, valueLength), value);

      result.emplace_back(key, value);

      if ((ampersandIndex == std::string::npos) || (ampersandIndex == parameterString.size())) {
        break;
      } else {
        currentIndex = ampersandIndex + 1;
      }
    }
  }
}

void SplitHttpParameters(const std::string &parameterString, std::map<std::string, std::string> &result) {
  std::vector<std::pair<std::string, std::string>> params;
  SplitHttpParameters(parameterString, params);

  for (const auto &pair : params) {
    result[pair.first] = pair.second;
  }
}

bool GenerateSslVerificationHosts(const std::string &originalHost, std::vector<std::string> &result) {
  if (IsHostAnIpAddress(originalHost)) {
    return false;
  }

  std::string host(originalHost);
  int64_t periods = std::count(host.begin(), host.end(), '.');

  while (periods >= 2) {
    result.emplace_back(host);
    host = "*" + host.substr(host.find('.'));
    result.emplace_back(host);
    host = host.substr(2);  // Get rid of leading "*."
    periods--;

    // We assume that the host name doesn't start with a .
    // We also assume that the TLD has no periods in it (i.e. com, tv) - we
    // would need a table of TLDs otherwise.
  }
  result.emplace_back(host);

  return true;
}

bool IsHostAnIpAddress(const std::string &hostName) {
  // Assume a standard 32-bit IPv4 address in dotted-decimal representation
  // (a.b.c.d).
  std::regex ipRegex("\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}", std::regex::ECMAScript);
  return std::regex_match(hostName, ipRegex);
}
