#include "core/threadedeventscheduler.hpp"

#include <assert.h>

ThreadedEventScheduler::ThreadedEventScheduler() : mState(EventSchedulerState::Running) {
  mThread = std::make_unique<std::thread>(std::bind(&ThreadedEventScheduler::EventSchedulerThreadProc, this));
}

ThreadedEventScheduler::~ThreadedEventScheduler() {
  assert(mState == EventSchedulerState::ShutDown);
  if (mState != EventSchedulerState::ShutDown) {
    Shutdown(nullptr);
    mThread->join();
  }
}

void ThreadedEventScheduler::EventSchedulerThreadProc() {
  while (mState != EventSchedulerState::ShutDown) {
    mEventQueue.WaitForEvent();
  }

  mEventQueue.Clear();

  if (mShutDownTask != nullptr) {
    mShutDownTask();
    mShutDownTask = nullptr;
  }
}

TaskId ThreadedEventScheduler::ScheduleTask(TaskParams&& taskParams) {
  if (mState != EventSchedulerState::Running) {
    return 0;
  }

  return mEventQueue.InsertTask(std::move(taskParams));
}

bool ThreadedEventScheduler::CancelTask(TaskId taskId) {
  if (mState != EventSchedulerState::Running) {
    return false;
  }

  return mEventQueue.RemoveTask(taskId);
}

bool ThreadedEventScheduler::Shutdown(TaskFunc&& shutdownTask) {
  EventSchedulerState expected = EventSchedulerState::Running;
  if (!mState.compare_exchange_strong(expected, EventSchedulerState::ShuttingDown)) {
    return false;
  }

  mEventQueue.InsertTask({[this, shutdownTask = std::move(shutdownTask)]() {
                            assert(mState == EventSchedulerState::ShuttingDown);
                            EventSchedulerState expectedState = EventSchedulerState::ShuttingDown;
                            if (mState.compare_exchange_strong(expectedState, EventSchedulerState::ShutDown)) {
                              mShutDownTask = shutdownTask;
                            }
                          },
    "Shutting down ThreadedEventScheduler"});

  return true;
}
