#include "core/time.hpp"

#include <assert.h>

#include <algorithm>
#include <limits>
#include <mutex>
#include <random>

namespace {
constexpr std::chrono::milliseconds kDefaultMaxInterval = std::chrono::milliseconds(1000 * 60);
constexpr std::chrono::milliseconds kDefaultJitter = std::chrono::milliseconds(1000 * 1);
}  // namespace

std::chrono::milliseconds JitterTime(std::chrono::milliseconds baseMs, std::chrono::milliseconds widthMs) {
  widthMs = std::min(widthMs, baseMs);

  if (widthMs != std::chrono::milliseconds(0)) {
    // NOTE: This has to be thread safe but also only created once for performance and seeing purposes.
    // Thus we have one static instance per thread.
    thread_local std::default_random_engine generator(GetSecondsSinceEpoch());
    std::uniform_int_distribution<int32_t> distribution(
      -static_cast<int32_t>(widthMs.count()), static_cast<int32_t>(widthMs.count()));
    int32_t jitter = distribution(generator);
    return std::chrono::milliseconds(static_cast<uint64_t>(static_cast<int32_t>(baseMs.count()) + jitter));
  } else {
    return baseMs;
  }
}

RetryBackoffTable::RetryBackoffTable() : mJitterMilliseconds(kDefaultJitter), mNextAttemptNumber(0) {
  CreateTable(kDefaultMaxInterval);
}

RetryBackoffTable::RetryBackoffTable(
  const std::vector<std::chrono::milliseconds> &tableMilliseconds, std::chrono::milliseconds retryJitterWidthMs)
    : mJitterMilliseconds(retryJitterWidthMs), mNextAttemptNumber(0) {
  mBackOffTableMilliseconds = tableMilliseconds;
  mJitterMilliseconds = retryJitterWidthMs;
}

RetryBackoffTable::RetryBackoffTable(
  std::chrono::milliseconds maxInterval, std::chrono::milliseconds retryJitterWidthMs)
    : mJitterMilliseconds(retryJitterWidthMs), mNextAttemptNumber(0) {
  CreateTable(maxInterval);
}

void RetryBackoffTable::Advance() {
  if (mNextAttemptNumber < mBackOffTableMilliseconds.size() - 1) {
    mNextAttemptNumber++;
  }
}

void RetryBackoffTable::Reset() {
  mNextAttemptNumber = 0;
}

std::chrono::milliseconds RetryBackoffTable::GetInterval() const {
  std::chrono::milliseconds jitter = mNextAttemptNumber == 0 ? std::chrono::milliseconds(0) : mJitterMilliseconds;
  return JitterTime(mBackOffTableMilliseconds[mNextAttemptNumber], jitter);
}

void RetryBackoffTable::CreateTable(std::chrono::milliseconds maxInterval) {
  // Compute the exponential table
  mBackOffTableMilliseconds.clear();

  std::chrono::milliseconds interval =
    std::chrono::milliseconds(std::min(static_cast<uint64_t>(1000), static_cast<uint64_t>(maxInterval.count())));

  // Don't let the table grow too large if someone put in a large or invalid max
  // interval
  while (interval < maxInterval && mBackOffTableMilliseconds.size() < 32) {
    mBackOffTableMilliseconds.push_back(interval);

    interval *= 2;
  }

  if (mBackOffTableMilliseconds.back() < maxInterval) {
    mBackOffTableMilliseconds.push_back(maxInterval);
  }
}
