#include "tracking/trackingapi.hpp"

#include "core/base64.hpp"
#include "core/httprequest.hpp"
#include "core/httprequestutils.hpp"
#include "core/stringutilities.hpp"
#include "core/time.hpp"
#include "json11/json11.hpp"

#include <assert.h>

#include <ios>
#include <sstream>

namespace {

constexpr char kSpadeUrl[] = "https://spade.twitch.tv/track/";

constexpr size_t kDefaultMaxBytesInBatch = 1024 * 500;  // 500 KB, limit for Spade.
constexpr uint32_t kDefaultMaxPendingEvents = 30;
constexpr std::chrono::seconds kDefaultFlushInterval = std::chrono::seconds(1);
constexpr std::chrono::seconds kFlushJitter = std::chrono::seconds(10);
constexpr std::chrono::minutes kMaxFlushBackoff = std::chrono::minutes(5);

size_t Base64ExpansionSize(size_t size) {
  // 4/3 for B64 expansion factor, +1 for floor
  return (size * 4 / 3) + 1;
}

size_t Base64ExpansionSizeWithArrayMarkup(size_t size) {
  // +2 for the json array []
  return Base64ExpansionSize(size + 2);
}
}  // namespace

TrackingAPI::TrackingAPI(ThreadedEventScheduler &scheduler)
    : mScheduler(scheduler),
      mBackoffTable(kMaxFlushBackoff, kFlushJitter),
      mMaxBytesInBatch(kDefaultMaxBytesInBatch),
      mMaxPendingEvents(kDefaultMaxPendingEvents),
      mFlushInterval(kDefaultFlushInterval),
      mFlushTaskId(0),
      mShutdown(false) {}

TrackingAPI::~TrackingAPI() {
  {
    std::lock_guard<std::mutex> am(mMutex);
    mShutdown = true;

    // Try and cancel the scheduled task
    if (mFlushTaskId != 0) {
      bool cancelled = mScheduler.CancelTask(mFlushTaskId);
      if (cancelled) {
        mFlushTaskId = 0;
      }
    }
  }

  // Wait for the pending task to complete
  for (;;) {
    {
      std::lock_guard<std::mutex> am(mMutex);
      if (mFlushTaskId == 0) {
        break;
      }
    }
    std::this_thread::sleep_for(std::chrono::milliseconds(100));
  }
}

std::string TrackingAPI::GetSpadeUrl() const {
  return kSpadeUrl;
}

void TrackingAPI::TrackEvent(TrackingEvent &&event) {
  // Serialize
  json11::Json json = event.ToJson();
  std::string serializedEvent = json.dump();

  // Make sure the single event is not too big - should never happen in practice
  size_t encodedEventSize = Base64ExpansionSizeWithArrayMarkup(serializedEvent.size());
  assert(encodedEventSize <= mMaxBytesInBatch);
  if (encodedEventSize > mMaxBytesInBatch) {
    // Just discard the event
    return;
  }

  {
    std::lock_guard<std::mutex> am(mMutex);

    // NOTE: We let the pending event list grow larger than the max number
    // configured but we'll throw them out if they fail to send
    mPendingEvents.emplace_back(std::move(serializedEvent));
  }

  ScheduleNextFlushIfNeeded(false);
}

void TrackingAPI::ScheduleNextFlushIfNeeded(bool isAfterFlush, bool flushSucceeded) {
  std::lock_guard<std::mutex> am(mMutex);

  if (isAfterFlush) {
    mFlushTaskId = 0;
  }

  // Shutting down so do nothing
  if (mShutdown) {
    return;
  }

  // Already scheduled
  if (mFlushTaskId != 0) {
    return;
  }

  // Nothing to flush
  if (mPendingEvents.empty()) {
    return;
  }

  // Determine how long to delay
  std::chrono::milliseconds delay;

  if (isAfterFlush && !flushSucceeded) {
    delay = mBackoffTable.GetInterval();
    mBackoffTable.Advance();
  } else {
    mBackoffTable.Reset();
    delay = JitterTime(mFlushInterval, kFlushJitter);
  }

  // Schedule the task to run
  mFlushTaskId = mScheduler.ScheduleTask(TaskParams(std::bind(&TrackingAPI::FlushEvents, this), delay));
}

void TrackingAPI::FlushEvents() {
  std::vector<std::string> inFlightEvents;
  std::string serialized;
  bool succeeded = true;

  {
    std::lock_guard<std::mutex> am(mMutex);

    if (!mPendingEvents.empty()) {
      std::stringstream stream;

      // Serialize and encode until we reach the max that can be sent at once
      stream << "[";

      size_t totalSerializedSize = 0;

      // Accumulate the encoded data until our max batch size is reached
      auto iter = mPendingEvents.begin();
      for (; iter != mPendingEvents.end(); ++iter) {
        const auto &serializedEvent = *iter;

        // Number of bytes if we were to immediately flush the event queue after
        // adding serializedEvent.
        size_t bytesToFlush =
          Base64ExpansionSizeWithArrayMarkup(serializedEvent.size() + totalSerializedSize + 1);  // +1 for the ,

        // Batch full
        if (bytesToFlush > mMaxBytesInBatch) {
          break;
        }

        totalSerializedSize += serializedEvent.size();

        // Append to the buffer
        if (iter != mPendingEvents.begin()) {
          stream << ",";
        }

        stream << serializedEvent;
      }

      // Terminate the serialized stream
      stream << "]";
      serialized = stream.str();

      // Make sure we have serialized correctly
#if defined(_DEBUG)
      {
        std::string err;
        json11::Json::parse(serialized, err);
        assert(err.empty());
      }
#endif

      // Add the events to the in-flight list and remove from pending
      inFlightEvents.swap(mPendingEvents);
    }
  }

  if (!inFlightEvents.empty()) {
    auto callback = [this, &succeeded, &inFlightEvents](
                      uint32_t statusCode, const std::vector<char> & /*body*/, void * /*userData*/) {
      std::lock_guard<std::mutex> am(mMutex);

      succeeded = Is2XX(statusCode);
      if (succeeded) {
        mBackoffTable.Reset();
        inFlightEvents.clear();
      }
      // A communication issue with the backend or server error
      else {
        // Failed to reach the server, possibly offline so save events for later
        if (statusCode == 0) {
          uint32_t numDropped = 0;
          EnqueueFailedEvents(numDropped, inFlightEvents);
        }
        // We got an HTTP status that is 3XX or 4XX so discard the events
        // now and pretend it succeeded
        else {
          succeeded = true;
        }
      }
    };

    std::string encodedBatch = Base64Encode(
      reinterpret_cast<const unsigned char *>(serialized.c_str()), static_cast<unsigned int>(serialized.length()));
    std::string requestBody = "data=" + encodedBatch;

    // Send the request synchronously
    bool sent = SendHttpRequest("Spade", kSpadeUrl, {HttpParam("Content-Type", "application/x-www-form-urlencoded")},
      reinterpret_cast<const uint8_t *>(requestBody.data()), requestBody.size(), HttpRequestType::Post, 5, nullptr,
      callback, nullptr);

    if (!sent) {
      succeeded = false;
    }
  }

  ScheduleNextFlushIfNeeded(true, succeeded);
}

void TrackingAPI::EnqueueFailedEvents(uint32_t &numDropped, std::vector<std::string> &inFlightEvents) {
  numDropped = 0;

  // Move the failed events back into the pending list
  mPendingEvents.insert(mPendingEvents.begin(), inFlightEvents.begin(), inFlightEvents.end());
  inFlightEvents.clear();

  // Too many events are queued up, drop some and notify
  if (static_cast<uint32_t>(mPendingEvents.size()) > mMaxPendingEvents) {
    // Drop the oldest events
    numDropped = static_cast<uint32_t>(mPendingEvents.size()) - mMaxPendingEvents;
    mPendingEvents.erase(mPendingEvents.begin(), mPendingEvents.begin() + static_cast<int>(numDropped));
  }
}
