import { parse } from 'query-string';
import { logger } from 'tachyon-logger';
import { Enum, flattenHeaderOrParam } from 'tachyon-utils';

export enum Mode {
  /**
   * The full Laser Array feature set which bootstaps Starshot and
   * manages error state automatically.
   */
  Full = 'full',
  /**
   * Disables most Laser Array functionality and goes directly
   * to the error page.  The retry button is shown and can be hooked
   * by native platforms to manage retries manually.
   */
  NativeError = 'native-error',
}

/**
 * Specifies which mode in which to run Laser Array.  If not specified 'full'
 * is assumed.
 *
 * Possible values:
 *
 *  full
 *    The full Laser Array feature set which bootstaps Starshot and
 *    manages error state automatically.
 *
 *  native-error
 *    Disables most Laser Array functionality and goes directly
 *    to the error page.  The retry button is shown and can be hooked
 *    by native platforms to manage retries manually.
 */
const MODE_PARAM_NAME = 'mode';

/**
 * Parses the `mode` query parameter.
 */
export const parseModeQueryParam = (): Mode => {
  const parsed = parse(window.location.search);
  const parsedMode = parsed[MODE_PARAM_NAME] ?? '';
  const mode = flattenHeaderOrParam(parsedMode) ?? Mode.Full;
  const value = Enum.convertValueFromExternal(Mode, mode);
  if (value) {
    return value;
  }
  logger.error({
    category: 'LaserArrayConfigRoot',
    message: `Unhandled config mode: ${mode}`,
    package: 'laser-array',
  });
  return Mode.Full;
};
