import { Platform } from 'tachyon-environment';
import { createLaserArrayAndroidNativeAppProxy } from './androidtv';
import { createLaserArrayNativeAppProxy } from './factory';
import type {
  LaserArrayNativeBrowserWindowProxy,
  LaserArrayNativeProxyType,
} from './proxy';
import { createLaserArrayStubNativeAppProxy } from './stub';

declare global {
  interface Window {
    androidTvNativeHooks: LaserArrayNativeBrowserWindowProxy | undefined;
  }
}

function mockAndroidNativeHooks(
  func: () => any,
): LaserArrayNativeBrowserWindowProxy {
  const mocks: LaserArrayNativeBrowserWindowProxy = {
    laserArrayErrorRetry: jest.fn(() => undefined),
    nativeProxyType: 'stub',
  };

  const previousHooks = window.androidTvNativeHooks;
  window.androidTvNativeHooks = mocks;
  func();
  window.androidTvNativeHooks = previousHooks;

  return mocks;
}

describe('native proxies', () => {
  const cases: { [value in Platform]: LaserArrayNativeProxyType } = {
    [Platform.Unknown]: 'stub',
    [Platform.MobileWeb]: 'stub',
    [Platform.Web]: 'stub',
    [Platform.Woodstock]: 'stub',
    [Platform.Prospero]: 'stub',
    [Platform.StarshotDev]: 'stub',
    [Platform.StarshotStaging]: 'stub',
    [Platform.StarshotWeb]: 'stub',
    [Platform.LG]: 'stub',
    [Platform.Samsung]: 'stub',
    [Platform.Vizio]: 'stub',
    [Platform.Vestel]: 'stub',
    [Platform.PS4]: 'stub',
    [Platform.Switch]: 'exit-url',
    [Platform.AndroidTV]: 'android-hooks',
    [Platform.FireTV]: 'android-hooks',
  };

  it.each(Object.entries(cases))(
    'uses proper proxy on platform ($s)',
    (platformValue, proxyType) => {
      const platform: Platform = platformValue as Platform;
      const proxy = createLaserArrayNativeAppProxy(platform);
      expect(proxy.nativeProxyType).toBe(proxyType);
    },
  );

  it('verifies StubNativeAppProxy is a no-op', () => {
    const proxy = createLaserArrayStubNativeAppProxy();

    expect(proxy.nativeProxyType).toBe('stub');

    expect(() => {
      proxy.laserArrayErrorRetry();
    }).not.toThrow();
  });

  it('verifies AndroidNativeAppProxy calls into expected native hooks', () => {
    const proxy = createLaserArrayAndroidNativeAppProxy();

    const nativeHooksMock = mockAndroidNativeHooks(() => {
      proxy.laserArrayErrorRetry();
    });

    expect(nativeHooksMock.laserArrayErrorRetry).toHaveBeenCalledTimes(1);
  });
});
