import type { FC } from 'react';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { Pageview } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import { LatencyTransitionComplete } from 'tachyon-latency-tracker';
import type { ButtonProps } from 'twitch-core-ui';
import {
  AlignItems,
  Button,
  ButtonSize,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Position,
  Title,
} from 'twitch-core-ui';
import { NINTENDO_SWITCH_ERROR_RETRY_URL } from '../../components';
import { LOADING_ERROR_ROUTENAME } from '../../config';
// eslint-disable-next-line import/no-internal-modules
import Error500 from '../../images/error-500.png';
import { useErrorPageHooks } from './useErrorPageHooks';

type ErrorPageProps = {
  message?: string;
};

// istanbul ignore next: trivial
const NintendoSwitchErrorRetryLink: FC = ({ children }) => (
  <a children={children} href={NINTENDO_SWITCH_ERROR_RETRY_URL} />
);
NintendoSwitchErrorRetryLink.displayName = 'NintendoSwitchErrorRetryLink';

// istanbul ignore next: trivial
const RetryButton: FC = () => {
  const { formatMessage } = useIntl();
  const { onRetryClicked } = useErrorPageHooks();
  const {
    common: { platform },
  } = useStaticEnvironment();

  // On Nintendo Switch we need to use a special URL that will close the browser with a result.
  // Unfortunately, for security reasons this cannot be done programmatically by accessing
  // window.location, it must be an anchor.  This means we need to generate slightly different
  // elements for the button for this case.
  const actionProps: ButtonProps =
    platform === Platform.Switch
      ? {
          linkTo: '/deferToRenderLink',
          renderLink: NintendoSwitchErrorRetryLink,
        }
      : {
          onClick: onRetryClicked,
        };

  return (
    <Button autoFocus size={ButtonSize.Large} tabIndex={1} {...actionProps}>
      {formatMessage('Retry', 'Retry')}
    </Button>
  );
};
RetryButton.displayName = 'RetryButton';

// istanbul ignore next: trivial
export const ErrorPage: FC<ErrorPageProps> = ({ message }) => {
  const { formatMessage } = useIntl();
  return (
    <>
      <LatencyTransitionComplete location={LOADING_ERROR_ROUTENAME} />
      <Pageview location={LOADING_ERROR_ROUTENAME} />
      <Layout
        alignItems={AlignItems.Center}
        attachBottom
        attachLeft
        attachRight
        attachTop
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        justifyContent={JustifyContent.Center}
        position={Position.Absolute}
      >
        <Title>
          {message ??
            formatMessage(
              'Looks like something went wrong, please try again later.',
              'InternalServerError',
            )}
        </Title>
        <Layout margin={{ top: 3 }}>
          <RetryButton />
        </Layout>
      </Layout>
      <Layout attachBottom attachLeft position={Position.Absolute}>
        <img alt="" src={Error500} />
      </Layout>
    </>
  );
};

ErrorPage.displayName = 'ErrorPage';
