import type { FC } from 'react';
import type { LogPayload } from 'tachyon-logger';
import { logger } from 'tachyon-logger';
import { Formats, exhaustedCase, fetchData, useAsync } from 'tachyon-utils';
import { Loading, Redirect } from '../../../components';
import { HEALTH_CHECK_URL, STARSHOT_LAUNCH_URL } from '../../../config';
import { ErrorPage } from '../../Error';

export async function healthcheck(): Promise<void> {
  try {
    await fetchData(HEALTH_CHECK_URL, undefined, Formats.TEXT);
  } catch (e: unknown) {
    const context: LogPayload['context'] = {
      healthCheckUrl: HEALTH_CHECK_URL,
    };
    if (e instanceof Response) {
      context.status = e.status;
      context.statusText = e.statusText;
    }
    reportError(e as Error, context);
    throw e;
  }
}

function reportError(error: Error, context: LogPayload['context']) {
  logger.error({
    category: 'healthcheck',
    context,
    error,
    level: 'error',
    message: error.message || 'Starshot healthcheck failed',
    package: 'laser-array',
  });
}

export const Loader: FC = () => {
  const state = useAsync(healthcheck);

  switch (state.status) {
    case 'pending':
      return <Loading />;
    case 'fulfilled':
      return <Redirect to={STARSHOT_LAUNCH_URL} />;
    case 'rejected':
      return <ErrorPage />;
    default: {
      return exhaustedCase(state, <ErrorPage />);
    }
  }
};

Loader.displayName = 'Loader';
