import { logger } from 'tachyon-logger';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { fetchData, useAsync } from 'tachyon-utils';
import { Loading, Redirect } from '../../../components';
import { HEALTH_CHECK_URL, STARSHOT_LAUNCH_URL } from '../../../config';
import { ErrorPage } from '../../Error';
import { Loader, healthcheck } from '.';

jest.mock('tachyon-utils', () => ({
  ...jest.requireActual('tachyon-utils'),
  fetchData: jest.fn(),
  useAsync: jest.fn(),
}));

jest.mock('tachyon-logger', () => ({ logger: { error: jest.fn() } }));

describe(Loader, () => {
  const mockUseAsync = useAsync as jest.Mock;
  const setup = createShallowWrapperFactory(Loader);

  beforeEach(() => {
    mockUseAsync.mockReset();
    mockUseAsync.mockReturnValue({
      error: undefined,
      status: 'pending',
      value: undefined,
    });
  });

  it('renders a loading state on first render', () => {
    const { wrapper } = setup();
    expect(wrapper.find(Loading)).toExist();
  });

  it('renders a redirect when the health check is successful', () => {
    mockUseAsync.mockReturnValue({
      error: undefined,
      status: 'fulfilled',
      value: undefined,
    });

    const { wrapper } = setup();
    expect(wrapper.find(Redirect)).toHaveProp({ to: STARSHOT_LAUNCH_URL });
  });

  it('renders the ErrorPage when the health check is unsuccessful', () => {
    mockUseAsync.mockReturnValue({
      error: undefined,
      status: 'rejected',
      value: undefined,
    });

    const { wrapper } = setup();
    expect(wrapper.find(ErrorPage)).toExist();
  });
});

describe(healthcheck, () => {
  const mockFetchData = fetchData as jest.Mock;
  const mockLogger = logger.error as jest.Mock;

  beforeEach(() => {
    mockLogger.mockReset();
  });

  it('logs the error when the network request fails', async () => {
    const networkError = new Error('Network request failed');
    mockFetchData.mockImplementationOnce(() => {
      throw networkError;
    });

    await expect(healthcheck()).rejects.toThrow();

    expect(mockLogger).toHaveBeenCalledWith({
      category: 'healthcheck',
      context: {
        healthCheckUrl: HEALTH_CHECK_URL,
      },
      error: networkError,
      level: 'error',
      message: 'Network request failed',
      package: 'laser-array',
    });
  });

  it('logs the error with status context when the server response is not ok', async () => {
    const serverResponse = new Response(null, {
      status: 301,
      statusText: 'Moved Permanently',
    });

    mockFetchData.mockImplementationOnce(() => {
      throw serverResponse;
    });

    await expect(healthcheck()).rejects.toEqual(serverResponse);

    expect(mockLogger).toHaveBeenCalledWith({
      category: 'healthcheck',
      context: {
        healthCheckUrl: HEALTH_CHECK_URL,
        status: 301,
        statusText: 'Moved Permanently',
      },
      error: serverResponse,
      level: 'error',
      message: 'Starshot healthcheck failed',
      package: 'laser-array',
    });
  });
});
