﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain
 proprietary information of Nintendo and/or its licensed developers and are
 protected by national and international copyright laws. They may not be
 disclosed to third parties or copied or duplicated in any form, in whole or in
 part, without the prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <curl/curl.h>
#include <nn/os.h>
#include <nn/ssl.h>

// [SKIP_SERVER_CERT_VERIFY]
// Please enable "#define SKIP_SERVER_CERT_VERIFY" to allow application to skip
// verification of the server certificate and its host name for Non-Release
// build. Below please find how to implement it in Non-Release build. Don't
// forget to include ssl_DebugApi.h header file. Let's suppose that all
// initialization is completed at this moment.
//
// - Call curl_debug_ssl_setopt() function and specify
// CURLDEBUG_OPT_SSL_ALLOW_DISABLE_VERIFY as
//   an option and "true" as an option value. Otherwise, the assertion will
//   terminate application upon curl_easy_setopt() call.
// - Call curl_easy_setopt() as described in libcurl documentation to configure
// the desired
//   verification options.
//
// Functions curl_debug_ssl_setopt() and curl_debug_ssl_getopt() are only
// introduced for debug purposes. IT IS PROHIBITED TO USE THESE FUNCTIONS IN A
// RELEASE BUILD!
//
// [SKIP_SERVER_CERT_VERIFY_PROD]
// Please enable "#define SKIP_SERVER_CERT_VERIFY_PROD" to allow application to
// skip verification of the server certificate and its host name for Release
// build. It is not recommended to do so unless there's a strong reason. Below
// please find how to implement it in Release build. Let's suppose that all
// initialization is completed at this moment.
//
// - Call curl_easy_setopt() function and specify
// CURLDEBUG_OPT_SSL_ALLOW_DISABLE_VERIFY as
//   an option and "true" as an option value. Otherwise, the assertion will
//   terminate application upon curl_easy_setopt() call.
// - Call curl_easy_setopt() as described in libcurl documentation to configure
// the desired
//   verification options.

// Enable this to allow SSL to skip verifying server's name and certificate are
// valid for Non-Release build.
//#define SKIP_SERVER_CERT_VERIFY

// Enable this to allow SSL to skip verifying server's name and certificate are
// valid for Release build.
//#define SKIP_SERVER_CERT_VERIFY_PROD

//---------------------------------------------------------------------------------------------
// HttpsHelperBase class
//---------------------------------------------------------------------------------------------
class HttpsHelperBase {
 public:
  typedef struct VerifyOption {
    bool isVerifyPeer;
    bool isVerifyName;
    bool isVerifyTime;
  } VerifyOption;

  HttpsHelperBase(){};
  ~HttpsHelperBase(){};
  int ConfigureVerifyOption(CURL *pInCurlHandle, VerifyOption *pInVerifyOption);

  static bool IsUrlHttps(const char *pInUrl);

 protected:
  bool GetSslResultByCurlInfo(nn::Result *pOutResult, CURL *pInCurlHandle, CURLINFO curlInfo);
  void PrintErrorMessage(CURL *pInCurlHandle, nn::Result *pInResult, CURLcode curlError);

  static nn::Result ImportPkiData(nn::ssl::Context *pSslContext);
};

//---------------------------------------------------------------------------------------------
// HttpsHelperForCallback class for CURLOPT_SSL_CTX_FUNCTION method
//---------------------------------------------------------------------------------------------
class HttpsHelperForCallback : public HttpsHelperBase {
 public:
  HttpsHelperForCallback();
  ~HttpsHelperForCallback();
  int Initialize(CURL *pInCurlHandle);
  void Finalize();
  void PrintErrorMessage(CURL *pInCurlHandle, CURLcode curlError);

  const char *GetServerCertBuffer() const;

  static const uint32_t VerifyResultArrayLen = 4;
  static const uint32_t ServerCertBufLen = 4 * 1024;

 private:
  nn::Result m_Result;  //<! nn::Result to store Result in CURLOPT_SSL_CTX_FUNCTION
  char *m_pServerCertBuf;

  static CURLcode CurlSslContextCallback(CURL *pCurl, void *pSslContext, void *pUserData);
  static size_t CurlSslConnectionBeforeFunction(CURL *pCurl, void *pSslConnection, void *pUserData) NN_NOEXCEPT;
  static size_t CurlSslConnectionAfterFunction(
    CURL *pCurl, void *pSslConnection, void *pUserData, curl_handshake_info handshakeInfo) NN_NOEXCEPT;
};

//---------------------------------------------------------------------------------------------
// HttpsHelperForCtxImport class for CURLOPT_SSL_CONTEXT method
//---------------------------------------------------------------------------------------------
class HttpsHelperForCtxImport : public HttpsHelperBase {
 public:
  HttpsHelperForCtxImport();
  ~HttpsHelperForCtxImport();
  int Initialize();
  void Finalize();
  int ImportSslContext(CURL *pInCurlHandle);
  void PrintErrorMessage(CURL *pInCurlHandle, CURLcode curlError);

 private:
  nn::ssl::Context *m_pSslContext;  //<! Pointer to the nn::ssl::Context shared
                                    // by multiple CURL handles
};
