﻿#include "app/graphics.hpp"

#include <EGL/egl.h>
#include <EGL/eglext.h>
#include <GL/glew.h>
#include <nn/nn_Assert.h>
#include <nn/vi.h>
#include <nv/nv_MemoryManagement.h>

#include <algorithm>
#include <cstdlib>

namespace {
constexpr int kMaxWidth = 1920;
constexpr int kMaxHeight = 1080;

EGLDisplay gEglDisplay = nullptr;
EGLSurface gEglSurface = nullptr;
EGLContext gEglContext = nullptr;

nn::vi::NativeWindowHandle gNativeWindowHandle = nullptr;
nn::vi::Display* gDisplay = nullptr;
nn::vi::Layer* gLayer = nullptr;

#pragma region Missing OpenGL Symbols

// OpenGL functions that need to be looked up at runtime
using GlClearFunc = void (*)(GLbitfield mask);
using GlClearColorFunc = void (*)(GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha);
using GlGetStringFunc = GLubyte* (*)(GLenum name);
using GlDrawArraysFunc = void* (*)(GLenum mode, GLint first, GLsizei count);
using GlGenTexturesFunc = void (*)(GLsizei n, GLuint* textures);
using GlBindTextureFunc = void (*)(GLenum target, GLuint texture);
using GlTexImage2DFunc = void (*)(GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height,
  GLint border, GLenum format, GLenum type, const void* pixels);
using GlTexParameteriFunc = void (*)(GLenum target, GLenum pname, GLint param);
using GlDeleteTexturesFunc = void(*)(GLsizei n, const GLuint* textures);

GlClearFunc gGlClear = nullptr;
GlClearColorFunc gGlClearColor = nullptr;
GlGetStringFunc gGlGetString = nullptr;
GlDrawArraysFunc gGlDrawArrays = nullptr;
GlGenTexturesFunc gGlGenTextures = nullptr;
GlBindTextureFunc gGlBindTexture = nullptr;
GlTexImage2DFunc gGlTexImage2D = nullptr;
GlTexParameteriFunc gGlTexParameteri = nullptr;
GlDeleteTexturesFunc gGlDeleteTextures = nullptr;

#pragma endregion

#pragma region Memory Allocation

void* NvAllocateFunction(size_t size, size_t alignment, void* /*userPtr*/) {
  return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}
void NvFreeFunction(void* addr, void* /*userPtr*/) {
  free(addr);
}
void* NvReallocateFunction(void* addr, size_t newSize, void* /*userPtr*/) {
  return realloc(addr, newSize);
}

void* NvDevtoolsAllocateFunction(size_t size, size_t alignment, void* /*userPtr*/) {
  return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}
void NvDevtoolsFreeFunction(void* addr, void* /*userPtr*/) {
  free(addr);
}
void* NvDevtoolsReallocateFunction(void* addr, size_t newSize, void* /*userPtr*/) {
  return realloc(addr, newSize);
}

#pragma endregion

void* GetProcAddress(const char* name) {
  return reinterpret_cast<void*>(::eglGetProcAddress(name));
}

template <typename T> T LookupOpenGlFunction(const char* name) {
  auto result = reinterpret_cast<T>(GetProcAddress(name));
  NN_ASSERT(result != nullptr, "%s not ready", name);
  return result;
}

}  // namespace

void InitializeGraphics() {
  // Setup GPU memory
  nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
  nv::SetGraphicsDevtoolsAllocator(
    NvDevtoolsAllocateFunction, NvDevtoolsFreeFunction, NvDevtoolsReallocateFunction, NULL);

  // NOTE: We use malloc here since the memory doesn't ever get freed according to Nintendo docs
  size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
  void* graphicsHeap = malloc(graphicsSystemMemorySize);
  nv::InitializeGraphics(graphicsHeap, graphicsSystemMemorySize);

  // Setup video interface system
  nn::vi::Initialize();

  nn::Result result = nn::vi::OpenDefaultDisplay(&gDisplay);
  NN_ASSERT(result.IsSuccess());

  result = nn::vi::CreateLayer(&gLayer, gDisplay, kMaxWidth, kMaxHeight);
  NN_ASSERT(result.IsSuccess());

  result = nn::vi::GetNativeWindow(&gNativeWindowHandle, gLayer);
  NN_ASSERT(result.IsSuccess());

  // Native Platform Graphics interface
  EGLBoolean eglResult;

  gEglDisplay = ::eglGetDisplay(EGL_DEFAULT_DISPLAY);
  NN_ASSERT(gEglDisplay != NULL, "eglGetDisplay failed.");

  eglResult = ::eglInitialize(gEglDisplay, 0, 0);
  NN_ASSERT(eglResult, "eglInitialize failed.");

  EGLint configAttribs[] = {
    EGL_RENDERABLE_TYPE, EGL_OPENGL_BIT,
    EGL_SURFACE_TYPE, EGL_WINDOW_BIT, EGL_RED_SIZE, 8, EGL_GREEN_SIZE, 8, EGL_BLUE_SIZE, 8, EGL_ALPHA_SIZE, 8,
    EGL_NONE};
  EGLint numConfigs = 0;
  EGLConfig config;
  eglResult = ::eglChooseConfig(gEglDisplay, configAttribs, &config, 1, &numConfigs);
  NN_ASSERT(eglResult && numConfigs == 1, "eglChooseConfig failed.");

  gEglSurface = ::eglCreateWindowSurface(gEglDisplay, config, static_cast<NativeWindowType>(gNativeWindowHandle), 0);
  NN_ASSERT(gEglSurface != EGL_NO_SURFACE, "eglCreateWindowSurface failed.");

  // Select OpenGL as the rendering API
  eglResult = eglBindAPI(EGL_OPENGL_API);
  NN_ASSERT(eglResult, "eglBindAPI failed.");

  // Create the graphics context
  EGLint contextAttribs[] = { EGL_CONTEXT_MAJOR_VERSION, 4, EGL_CONTEXT_MINOR_VERSION, 5, EGL_NONE };
  gEglContext = ::eglCreateContext(gEglDisplay, config, EGL_NO_CONTEXT, contextAttribs);
  NN_ASSERT(gEglContext != EGL_NO_CONTEXT, "eglCreateContext failed. %d", eglGetError());

  eglResult = ::eglMakeCurrent(gEglDisplay, gEglSurface, gEglSurface, gEglContext);
  NN_ASSERT(eglResult, "eglMakeCurrent failed.");

  eglResult = ::eglSwapInterval(gEglDisplay, 1);
  NN_ASSERT(eglResult, "eglSwapInterval failed.");

  ::glewInit();

  // On NX with GLEW, OpenGL APIs that are defined in OpenGL 1.x need to be queried with eglGetProcAddress
  gGlClear = LookupOpenGlFunction<GlClearFunc>("glClear");
  gGlClearColor = LookupOpenGlFunction<GlClearColorFunc>("glClearColor");
  gGlGetString = LookupOpenGlFunction<GlGetStringFunc>("glGetString");
  gGlDrawArrays = LookupOpenGlFunction<GlDrawArraysFunc>("glDrawArrays");
  gGlGenTextures = LookupOpenGlFunction<GlGenTexturesFunc>("glGenTextures");
  gGlBindTexture = LookupOpenGlFunction<GlBindTextureFunc>("glBindTexture");
  gGlTexImage2D = LookupOpenGlFunction<GlTexImage2DFunc>("glTexImage2D");
  gGlTexParameteri = LookupOpenGlFunction<GlTexParameteriFunc>("glTexParameteri");
  gGlDeleteTextures = LookupOpenGlFunction<GlDeleteTexturesFunc>("glDeleteTextures");
}

void ShutdownGraphics() {
  EGLBoolean eglResult;

  eglResult = ::eglMakeCurrent(gEglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
  NN_ASSERT(eglResult, "eglMakeCurrent failed.");
  eglResult = ::eglTerminate(gEglDisplay);
  NN_ASSERT(eglResult, "eglTerminate failed.");
  eglResult = ::eglReleaseThread();
  NN_ASSERT(eglResult, "eglReleaseThread failed.");

  nn::vi::DestroyLayer(gLayer);
  nn::vi::CloseDisplay(gDisplay);
  nn::vi::Finalize();

  nv::FinalizeGraphics();
}

void SwapBuffers() {
  ::eglSwapBuffers(gEglDisplay, gEglSurface);
}

#pragma region Missing OpenGL Symbols

// NOTE: These are here to simplify using these functions that are looked up at runtime and
// missing at link-time
void GLAPIENTRY glClear(GLbitfield mask) {
  NN_ASSERT(gGlClear != nullptr, "gGlClear not ready");
  gGlClear(mask);
}

void GLAPIENTRY glClearColor(GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha) {
  NN_ASSERT(gGlClearColor != nullptr, "gGlClearColor not ready");
  gGlClearColor(red, green, blue, alpha);
}

void GLAPIENTRY glDrawArrays(GLenum mode, GLint first, GLsizei count) {
  NN_ASSERT(gGlDrawArrays != nullptr, "gGlDrawArrays not ready");
  gGlDrawArrays(mode, first, count);
}

void GLAPIENTRY glGenTextures(GLsizei n, GLuint* textures) {
  NN_ASSERT(gGlGenTextures != nullptr, "gGlGenTextures not ready");
  gGlGenTextures(n, textures);
}

void GLAPIENTRY glBindTexture(GLenum target, GLuint texture) {
  NN_ASSERT(gGlBindTexture != nullptr, "gGlBindTexture not ready");
  gGlBindTexture(target, texture);
}

void GLAPIENTRY glTexImage2D(GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height,
  GLint border, GLenum format, GLenum type, const void* pixels) {
  NN_ASSERT(gGlTexImage2D != nullptr, "gGlTexImage2D not ready");
  gGlTexImage2D(target, level, internalformat, width, height, border, format, type, pixels);
}

void GLAPIENTRY glTexParameteri(GLenum target, GLenum pname, GLint param) {
  NN_ASSERT(gGlTexParameteri != nullptr, "gGlTexParameteri not ready");
  gGlTexParameteri(target, pname, param);
}

void GLAPIENTRY glDeleteTextures(GLsizei n, const GLuint* textures) {
  NN_ASSERT(gGlDeleteTextures != nullptr, "gGlDeleteTextures not ready");
  gGlDeleteTextures(n, textures);
}

#pragma endregion 
