#include "app/graphics.hpp"
#include "app/ipc.hpp"
#include "app/net.hpp"
#include "app/offline.hpp"
#include "app/online.hpp"
#include "app/save.hpp"
#include "app/splash.hpp"
#include "app/ugc.hpp"
#include "app/utility.hpp"
#include "core/httprequest.hpp"
#include "core/signal.hpp"
#include "core/threadedeventscheduler.hpp"
#include "tracking/trackingapi.hpp"
#include "tracking/trackingevents.hpp"

#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/oe.h>
#include <nn/oe/oe_ApplicationControlTypes.h>
#include <nn/settings/settings_Language.h>
#include <nn/time/time_Api.h>
#include <nn/util/util_ScopeExit.h>

#include <cassert>
#include <string>

using namespace app;

namespace {
// clang-format off
constexpr char kLaserArrayUrl[] = "laserarray.htdocs/index.html#mode=native-error";
// clang-format on

ThreadedEventScheduler gBackgroundScheduler;
std::unique_ptr<TrackingAPI> gSpade;
TwitchSaveData gSaveData;
std::unique_ptr<std::thread> gEventThread;          // Thread to handle events from the OS.
std::atomic<nn::oe::OperationMode> gOperationMode;  // Docked or handheld
std::atomic_bool gExit(false);                      // Whether or not to exit the app.
std::vector<uint8_t> gRomBuffer;

void InitializeFilesystem() {
  size_t cacheSize = 0;
  auto result = nn::fs::QueryMountRomCacheSize(&cacheSize);
  NN_ASSERT(result.IsSuccess(), "QueryMountRomCacheSize failed");

  gRomBuffer.resize(cacheSize);

  result = nn::fs::MountRom("Asset", gRomBuffer.data(), cacheSize);
  NN_ASSERT(result.IsSuccess(), "MountRom failed");
}

void ShutdownFilesystem() {
  nn::fs::Unmount("Asset");
}

void InitializeTracking() {
  if (gSpade != nullptr) {
    return;
  }

  gSpade = std::make_unique<TrackingAPI>(gBackgroundScheduler);
}

void TrackEvent(TrackingEvent&& event) {
  if (gSpade == nullptr) {
    return;
  }

  gSpade->TrackEvent(std::move(event));
}

void ShutdownTracking() {
  gSpade.reset();
}

void ShutdownScheduler() {
  auto waitUntilAtLatest = std::chrono::steady_clock::now() + std::chrono::seconds(5);
  gBackgroundScheduler.Shutdown({});
  while (gBackgroundScheduler.GetState() != ThreadedEventScheduler::EventSchedulerState::ShutDown &&
         std::chrono::steady_clock::now() < waitUntilAtLatest) {
    std::this_thread::sleep_for(std::chrono::milliseconds(100));
  }
}

void SetTrackingProperties(TrackingEvent& event) {
  nn::settings::LanguageCode language = nn::oe::GetDesiredLanguage();

  event.clientApp = "switch";
  event.language = language.string;
  event.osName = "switch";
  event.osVersion = "";  // There is no way to get this in a production build
  event.twitchAppVersion = GetApplicationVersion();
  event.twitchDeviceId = GetDeviceId();
  event.twitchPlatform = "switch";
}

void HandleStarshotCallback(const OnlineWebExitData& result) {
  auto iter = result.exitCallbackParameters.find("callback");
  if (iter == result.exitCallbackParameters.end()) {
    // We expect a parameter indicating the reason for the callback
    return;
  }

  // Login event
  if (iter->second == "login") {
    auto oauthTokenIter = result.exitCallbackParameters.find("access-token");
    if (oauthTokenIter != result.exitCallbackParameters.end()) {
      gSaveData.oauthToken = oauthTokenIter->second;
    }

    auto refreshTokenIter = result.exitCallbackParameters.find("refresh-token");
    if (refreshTokenIter != result.exitCallbackParameters.end()) {
      gSaveData.refreshToken = refreshTokenIter->second;
    }

    SaveData(gSaveData);

    NativeShellLoginTrackingEvent loginEvent;
    SetTrackingProperties(loginEvent);
    TrackEvent(std::move(loginEvent));
  }
  // Logout event
  else if (iter->second == "logout") {
    gSaveData.oauthToken = "";
    gSaveData.refreshToken = "";

    SaveData(gSaveData);

    NativeShellLogoutTrackingEvent logoutEvent;
    SetTrackingProperties(logoutEvent);
    TrackEvent(std::move(logoutEvent));
  }
}

bool CheckIfTwitchIsReachable() {
  bool reachable = false;
  Signal signal;
  // Send the request on the background scheduler since we currently don't have CURL set up to
  // run on different threads with the same context
  gBackgroundScheduler.ScheduleTask({[&reachable, &signal]() {
    auto succeeded = SendHttpRequest(
      "GET", "https://switch.tv.twitch.tv/_debug/running", {}, nullptr, 0, HttpRequestType::Get, 5,
      [&reachable](uint32_t statusCode, const HeaderMap& /*headers*/, void* /*userData*/) {
        reachable = statusCode > 0;
        return false;
      },
      nullptr, nullptr);
    if (!succeeded) {
      reachable = false;
    }
    signal.Fire();
  }});
  signal.Wait();
  return reachable;
}

bool HandleSystemEvent(nn::oe::Message& message) {
  switch (message) {
    case nn::oe::MessageExitRequest:
      return true;
    case nn::oe::MessageOperationModeChanged:
      gOperationMode = nn::oe::GetOperationMode();
      break;
    case nn::oe::MessagePerformanceModeChanged:
      // Performance mode (CPU normal or boost) currently doesn't affect our app directly
      break;
    default:
      break;
  }

  return false;
}

void StartSystemMessageProcessingThread() {
  gEventThread = std::make_unique<std::thread>([]() {
    while (!gExit) {
      nn::oe::Message message = nn::oe::PopNotificationMessage();
      gExit = HandleSystemEvent(message);
    }
    nn::oe::LeaveExitRequestHandlingSection();
    nn::oe::SetPerformanceModeChangedNotificationEnabled(false);
    nn::oe::SetOperationModeChangedNotificationEnabled(false);
  });
  nn::oe::EnterExitRequestHandlingSection();
  nn::oe::SetPerformanceModeChangedNotificationEnabled(true);
  nn::oe::SetOperationModeChangedNotificationEnabled(true);
}

}  // namespace

extern "C" void nnMain() NN_NOEXCEPT {
  nn::oe::Initialize();
  nn::time::Initialize();
  nn::account::Initialize();

  gOperationMode = nn::oe::GetOperationMode();

  InitializeFilesystem();

  // Schedule the Nintendo splash screen to be hidden
  gBackgroundScheduler.ScheduleTask({[]() {
                                       InitializeGraphics();
                                       InitializeTwitchSplashScreen();
                                       nn::oe::FinishStartupLogo();
                                       std::this_thread::sleep_for(std::chrono::milliseconds(400));
                                       ShowTwitchSplashScreen();
                                     },
    std::chrono::milliseconds(1000)});

  StartSystemMessageProcessingThread();

  InitializeHttp();

  // We always need to initialize the network in native in order to perform the NSA ID check
  GoOnline();

  // Nintendo has a requirement that if you use the network at all with your own independent server that you have to
  // ask the system to provide you with a Nintendo Service ID token which verifies that the user is allowed to use the
  // network. This is the case even if the requests you're making are the same as the ones that are used in the
  // WebApplet. But if Nintendo denies your usage of the network due to the NSA ID token not being available we can just
  // continue and launch the WebApplet as long as we don't use the network from native.  This means no metrics and we have 
  // to assume the reachability check for our backend has succeeded.
  // 
  // We could do a lot more sophistocated online state management.  If the app is launched in airplane mode we will likely fail
  // to get an NSA ID token and this prevents future native networking.  We could technically try and get one the next time we have a
  // valid network connection.  But it's not worth it right now since all we do is send a handful of edge case metrics events 
  // and do a reachability check only in the case that the user has a linked Nintendo account.
  bool hasNsaAccount = IsUserNintendoServiceAccountAvailable();
  if (hasNsaAccount) {
    InitializeTracking();
  }

  NativeShellLaunchTrackingEvent launchEvent;
  SetTrackingProperties(launchEvent);
  TrackEvent(std::move(launchEvent));

  // Try and load credentials
  LoadData(gSaveData);

  const std::string deviceId = GetDeviceId();

  // TODO: If we use NetFront then we should be calling these on each visit to the channel page where chat shows up.
  bool chattingAllowed = StartChatting();
  std::string context;
  bool firstLoad = true;
  Signal hideTwitchSplashScreenSignal;

  while (!gExit) {
    std::string starshotUrl =
      GenerateStarshotUrl(deviceId, gSaveData.oauthToken, gSaveData.refreshToken, chattingAllowed, context);

    if (firstLoad) {
      firstLoad = false;

      // Schedule the Twitch splash screen to be shown for a bit before loading the webview
      gBackgroundScheduler.ScheduleTask(
        {[&hideTwitchSplashScreenSignal]() { hideTwitchSplashScreenSignal.Fire(); }, std::chrono::milliseconds(2000)});
    }

    // Check if we can contact Twitch
    bool twitchReachable = GoOnline();
    if (hasNsaAccount) {
      twitchReachable = twitchReachable && CheckIfTwitchIsReachable();
    }

    hideTwitchSplashScreenSignal.Wait();

    if (twitchReachable) {
      const auto onlineExitData = DisplayOnlineWebPage(starshotUrl);

      // Clear the last context since it's specific to the callback from Starshot that was last handled
      context.clear();

      switch (onlineExitData.reason) {
        case nn::web::WebExitReason::WebExitReason_CallbackUrlReached: {
          // Capture the last context
          auto iter = onlineExitData.exitCallbackParameters.find("context");
          if (iter != onlineExitData.exitCallbackParameters.end()) {
            context = iter->second;
          }
          // Process the callback from Starshot
          HandleStarshotCallback(onlineExitData);
          break;
        }
        case nn::web::WebExitReason::WebExitReason_NetworkConnectionFailed:
        case nn::web::WebExitReason::WebExitReason_ExitMessage:
        case nn::web::WebExitReason::WebExitReason_EndButtonPressed:
        case nn::web::WebExitReason::WebExitReason_BackButtonPressed:
        default: {
          break;
        }
      }
    } else {
      DisplayOfflineWebPage(kLaserArrayUrl);
    }
  }

  EndChatting();

  ShutdownTracking();
  ShutdownHttp();
  ShutdownTwitchSplashScreen();
  ShutdownGraphics();
  ShutdownScheduler();
  ShutdownFilesystem();

  nn::time::Finalize();
}
