﻿#include "app/net.hpp"
#include "app/utility.hpp"

#include "app/curlhelper.hpp"
#include "core/httprequest.hpp"

#include <curl/curl.h>
#include <nn/init.h>
#include <nn/nifm.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <stdio.h>

#include <sstream>

namespace {
bool gNifmInitialized = false;
bool gSocketsInitialized = false;
nn::socket::ConfigDefaultWithMemory gSocketConfigWithMemory;
CURL *gCurl = nullptr;

struct ReadData {
  ReadData() : bytesConsumed(0) {}

  const uint8_t *requestBody;
  size_t requestBodySize;

  size_t bytesConsumed;
};

// Write Body
size_t CurlWrite_CallbackFunc(void *contents, size_t size, size_t nmemb, std::vector<char> *v) {
  size_t length = size * nmemb;
  v->insert(v->end(), (char *)contents, (char *)contents + length);
  return length;
}

// Read response body
size_t CurlRead_CallbackFunc(char *buffer, size_t size, size_t nitems, ReadData *readData) {
  size_t bytesLeft = readData->requestBodySize - readData->bytesConsumed;
  size_t bytesRequested = size * nitems;
  size_t bytesToRead = std::min(bytesLeft, bytesRequested);

  memcpy(buffer, readData->requestBody + readData->bytesConsumed, bytesToRead);
  readData->bytesConsumed += bytesToRead;

  return bytesToRead;
}

// Write headers
size_t CurlHeader_CallbackFunc(char *buffer, size_t size, size_t nitems, HeaderMap *headerMap) {
  size_t bytesRequested = size * nitems;

  std::string line;
  line.assign(buffer, bytesRequested);
  auto colonPos = line.find(':');

  if (colonPos != std::string::npos && colonPos > 0 && colonPos + 1 <= line.length()) {
    (*headerMap)[line.substr(0, colonPos)] = line.substr(colonPos + 2, line.size());
  }

  return bytesRequested;
}
}  // namespace

std::string app::PrepareCallbackUrlData(const std::string &callbackUrl, const std::string &callbackPrefix) {
  if (callbackUrl.find(callbackPrefix) == 0) {
    return callbackUrl.substr(callbackPrefix.size());
  } else {
    return "";
  }
}

bool app::InitializeHttp() {
  if (gCurl) {
    return false;
  }

  CURLcode res = CURLE_OK;
  nn::Result ret = nn::Result();

  /* since the following will utilize the system network interface, we must
     initialize network interface manager (NIFM) */
  if (!gNifmInitialized) {
    ret = nn::nifm::Initialize();
    if (ret.IsSuccess()) {
      gNifmInitialized = true;
    } else {
      return false;
    }
  }

  /* Initialize socket library, while supplying configuration and memory pool */
  if (!gSocketsInitialized) {
    ret = nn::socket::Initialize(gSocketConfigWithMemory);

    if (ret.IsSuccess()) {
      gSocketsInitialized = true;
    } else {
      NN_LOG("\n[ERROR] nn::socket::Initialize() failed. Err Desc: %d\n\n", ret.GetDescription());
      return false;
    }
  }

  /* initialize using system malloc for libcurl for testing */
  if (gCurl == nullptr) {
    res = curl_global_init(CURL_GLOBAL_DEFAULT);
    if (res != CURLE_OK) {
      NN_LOG("\n[ERROR] curl_global_init failed. Err: %d\n\n", res);
      curl_global_cleanup();
      return false;
    }

    gCurl = curl_easy_init();

    if (gCurl == nullptr) {
      curl_global_cleanup();
      return false;
    }
  }

  return true;
}

bool app::GoOnline() {
  nn::nifm::CancelNetworkRequest();
  
  // Request usage of the network interface asynchronously
  nn::nifm::SubmitNetworkRequest();

  // Poll the status of the request
  while (nn::nifm::IsNetworkRequestOnHold()) {
    NN_LOG("Waiting for network interface availability...\n");
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(250));
  }

  // If this succeededs then the device has successfully pinged a Nintendo server
  if (nn::nifm::IsNetworkAvailable()) {
    NN_LOG("Got access to network interface.\n");
    return true;
  } else {
    NN_LOG("Request for access to network interface denied.\n");
    nn::nifm::CancelNetworkRequest();
    return false;
  }
}

bool app::IsUserNintendoServiceAccountAvailable() {
  bool available = false;

  UseUserHandle([&available](nn::account::UserHandle userHandle, nn::account::Uid userId) {
    nn::Result result = nn::account::IsNetworkServiceAccountAvailable(&available, userHandle);
    available = available && result.IsSuccess();

    if (!available) {
      return;
    }

    nn::account::AsyncContext context;
    result = nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync(&context, userHandle);
    if (result.IsFailure()) {
      available = false;
      return;
    }
    
    // Wait synchronously for the async request to finish
    nn::os::SystemEvent systemEvent;
    result = context.GetSystemEvent(&systemEvent);
    if (result.IsFailure()) {
      available = false;
      return;
    }

    systemEvent.Wait();

    available = context.GetResult().IsSuccess();
  });

  // According to 0134 we don't have to show a message for this if we can safely continue without sending network
  // requests
  // https://developer.nintendo.com/html/online-docs/g1kr9vj6-en/Packages/Docs/Guidelines/NX-Guidelines/contents/Pages/Page_100905539.html#GuidelineDescLabel_220074733

  return available;
}

bool app::ShutdownHttp() {
  if (gCurl == nullptr) {
    return false;
  }

  /* always cleanup */
  curl_easy_cleanup(gCurl);

  curl_global_cleanup();
  nn::socket::Finalize();
  nn::nifm::CancelNetworkRequest();

  gCurl = nullptr;

  return true;
}

// NOTE: This is declared in the core library but defined here so it is found at link-time.
bool SendHttpRequest(const std::string & /*requestName*/, const std::string &url,
  const std::vector<HttpParam> &requestHeaders, const uint8_t *requestBody, size_t requestBodySize,
  HttpRequestType httpReqType, uint32_t timeOutInSecs, const HttpRequestHeadersCallback &headersCallback,
  const HttpRequestCallback &responseCallback, void *userData) {
  if (gCurl == nullptr) {
    return false;
  }

  if (url.empty()) {
    return false;
  }

  curl_easy_setopt(gCurl, CURLOPT_URL, url.c_str());

  // Support https
  // TODO: Do it only once instead of per-request?
  HttpsHelperForCallback httpsHelper;
  if (HttpsHelperBase::IsUrlHttps(url.c_str())) {
    HttpsHelperForCallback::VerifyOption verifyOption;

    if (httpsHelper.Initialize(gCurl) < 0) {
      return false;
    }

    verifyOption.isVerifyPeer = true;
    verifyOption.isVerifyName = true;
    verifyOption.isVerifyTime = false;
    if (httpsHelper.ConfigureVerifyOption(gCurl, &verifyOption) < 0) {
      return false;
    }
  }

  curl_easy_setopt(gCurl, CURLOPT_FOLLOWLOCATION, 1L);

  // default to http get
  curl_easy_setopt(gCurl, CURLOPT_POST, 0);
  curl_easy_setopt(gCurl, CURLOPT_HTTPGET, 1);
  curl_easy_setopt(gCurl, CURLOPT_CUSTOMREQUEST, nullptr);

  switch (httpReqType) {
    case HttpRequestType::Put:
      curl_easy_setopt(gCurl, CURLOPT_CUSTOMREQUEST, "PUT");
      break;
    case HttpRequestType::Post:
      curl_easy_setopt(gCurl, CURLOPT_POST, 1);
      break;
    case HttpRequestType::Delete:
      curl_easy_setopt(gCurl, CURLOPT_CUSTOMREQUEST, "DELETE");
      break;
    case HttpRequestType::Get:
    default:
      break;
  }

  struct CurlListProtector {
    CurlListProtector(const std::vector<HttpParam> &requestHeaders) {
      for (const auto &requestHeader : requestHeaders) {
        headers = curl_slist_append(headers, (requestHeader.paramName + requestHeader.paramValue).c_str());
      }
    }

    ~CurlListProtector() {
      if (headers) {
        curl_slist_free_all(headers);
      }
    }

    struct curl_slist *headers = nullptr;
  };

  CurlListProtector headersList(requestHeaders);

  curl_easy_setopt(gCurl, CURLOPT_HTTPHEADER, headersList.headers);
  curl_easy_setopt(gCurl, CURLOPT_TIMEOUT, timeOutInSecs);
  curl_easy_setopt(gCurl, CURLOPT_WRITEFUNCTION, CurlWrite_CallbackFunc);

  std::vector<char> responseData;
  curl_easy_setopt(gCurl, CURLOPT_WRITEDATA, &responseData);

  HeaderMap headerMap;

  ReadData readData;
  readData.requestBody = requestBody;
  readData.requestBodySize = requestBodySize;

  if (requestBody != nullptr && requestBodySize > 0) {
    curl_easy_setopt(gCurl, CURLOPT_READDATA, &readData);
    curl_easy_setopt(gCurl, CURLOPT_READFUNCTION, CurlRead_CallbackFunc);
    curl_easy_setopt(gCurl, CURLOPT_POSTFIELDSIZE, readData.requestBodySize);
  } else {
    curl_easy_setopt(gCurl, CURLOPT_READDATA, nullptr);
    curl_easy_setopt(gCurl, CURLOPT_READFUNCTION, nullptr);
    curl_easy_setopt(gCurl, CURLOPT_POSTFIELDSIZE, 0);
  }

  curl_easy_setopt(gCurl, CURLOPT_HEADERFUNCTION, &CurlHeader_CallbackFunc);
  curl_easy_setopt(gCurl, CURLOPT_HEADERDATA, &headerMap);

#ifdef NN_SDK_BUILD_DEBUG
  curl_easy_setopt(gCurl, CURLOPT_VERBOSE, 1);
#endif
  CURLcode res = curl_easy_perform(gCurl);

  if (res == CURLE_OK) {
    long responseCode;
    curl_easy_getinfo(gCurl, CURLINFO_RESPONSE_CODE, &responseCode);

    bool wantResponse = true;
    if (headersCallback != nullptr) {
      wantResponse = headersCallback(responseCode, headerMap, userData);
    }

    if (wantResponse && responseCallback != nullptr) {
      responseCallback(responseCode, responseData, userData);
    }
  }

  return true;
}
