﻿#include "app/offline.hpp"

#include "app/ipc.hpp"
#include "app/net.hpp"
#include "app/utility.hpp"
#include "core/httprequestutils.hpp"

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/web.h>
#include <stdint.h>

#include <string>

namespace {
using namespace app;

void HandleResult(const nn::web::OfflineHtmlPageReturnValue& offlinePageReturnValue, OfflineWebExitData& result) {
  result.reason = offlinePageReturnValue.GetOfflineExitReason();

  if (result.reason == nn::web::OfflineExitReason_CallbackUrlReached) {
    const char* lastUrl = offlinePageReturnValue.GetLastUrl();
    if (lastUrl != nullptr) {
      result.exitUrl = lastUrl;
    }

    std::string params = PrepareCallbackUrlData(result.exitUrl, kLaserArrayIpcCallbackUrlPrefix);
    SplitHttpParameters(params, result.exitCallbackParameters);

    NN_LOG("Offline callback received: '%s'", params.c_str());
  }
}

}  // namespace

app::OfflineWebExitData app::DisplayOfflineWebPage(const std::string& path) NN_NOEXCEPT {
  OfflineWebExitData result;

  nn::web::ShowOfflineHtmlPageArg showOfflinePageArg(path.c_str());
  showOfflinePageArg.SetFooterEnabled(false);
  showOfflinePageArg.SetJsExtensionEnabled(true);
  showOfflinePageArg.SetPointerEnabled(false);
  showOfflinePageArg.SetTouchEnabledOnContents(true);
  showOfflinePageArg.SetPageScrollIndicatorEnabled(false);

  // Configure callback url
  nn::web::OfflineHtmlPageReturnValue offlinePageReturnValue;

  // Blocking call to open the webview
  nn::Result showOfflineResult = nn::web::ShowOfflineHtmlPage(&offlinePageReturnValue, showOfflinePageArg);
  if (showOfflineResult.IsSuccess()) {
    HandleResult(offlinePageReturnValue, result);
    NN_LOG("Exit Reason %d", result.reason);

    switch (result.reason) {
      case nn::web::OfflineExitReason_CallbackUrlReached: {
        break;
      }
      case nn::web::OfflineExitReason_EndButtonPressed: {
        break;
      }
      case nn::web::OfflineExitReason_BackButtonPressed: {
        break;
      }
      case nn::web::OfflineExitReason_ExitMessage: {
        break;
      }
      case nn::web::OfflineExitReason_LastWindowDeleted: {
        break;
      }
      case nn::web::OfflineExitReason_LocalNetworkDisconnectionDenied: {
        break;
      }
      case nn::web::OfflineExitReason_MediaPlayerClosed: {
        break;
      }
      case nn::web::OfflineExitReason_CausedByWebPage: {
        break;
      }
      case nn::web::OfflineExitReason_CallerDataCorrupted: {
        break;
      }
      case nn::web::OfflineExitReason_Unexpected: {
        break;
      }
      default: {
      }
    }
  } else {
    NN_LOG("Cancel");

    result.reason = nn::web::OfflineExitReason_Unexpected;
  }

  return result;
}
