#include "app/online.hpp"

#include "app/ipc.hpp"
#include "app/net.hpp"
#include "app/utility.hpp"
#include "core/httprequestutils.hpp"

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/web.h>
#include <stdint.h>

#include <string>

namespace {
using namespace app;

// clang-format off
#if !TTV_PUBLISH
 constexpr char kStarshotUrl[] = "https://87hn6uil7zv5hvmly5mz.tv.twitch.a2z.com"; // HEAD of main branch
// constexpr char kStarshotUrl[] = "https://asdfasdfasdfasdfasdfasdfasdfasd.com"; // Invalid host
// constexpr char kStarshotUrl[] = "https://fix7y1uvk4b1ahet92bx.tv.twitch.a2z.com";
// constexpr char kStarshotUrl[] = "https://j7738ebzthft3lh5e2g0.tv.twitch.a2z.com";
// constexpr char kStarshotUrl[] = "https://osposbc8hid4yfli2i3e.tv.twitch.a2z.com/";
// constexpr char kStarshotUrl[] = "https://87hn6uil7zv5hvmly5mz.tv.twitch.a2z.com/bufferbunny"; // HEAD of main branch, no ads
// constexpr char kStarshotUrl[] = "https://87hn6uil7zv5hvmly5mz.tv.twitch.a2z.com/adsbunny"; // HEAD of main branch, always preroll
// constexpr char kStarshotUrl[] = "https://87hn6uil7zv5hvmly5mz.tv.twitch.a2z.com/twitch_platforms_live_6"; // HEAD of main branch
// constexpr char kStarshotUrl[] = "https://mrohkkcptggjkje1.switch.tv.twitch.tv"; // Prod
 #else
constexpr char kStarshotUrl[] = "https://mrohkkcptggjkje1.switch.tv.twitch.tv";  // Prod
#endif
 // clang-format on

 void HandleResult(const nn::web::WebPageReturnValue& onlinePageReturnValue, OnlineWebExitData& result) {
  result.reason = onlinePageReturnValue.GetWebExitReason();

  if (result.reason == nn::web::WebExitReason_CallbackUrlReached) {
    const char* lastUrl = onlinePageReturnValue.GetLastUrl();
    if (lastUrl != nullptr) {
      result.exitUrl = lastUrl;
    }

    std::string params = PrepareCallbackUrlData(result.exitUrl, kStarshotIpcCallbackUrlPrefix);
    SplitHttpParameters(params, result.exitCallbackParameters);

    NN_LOG("Online callback received: '%s'", params.c_str());
  }
}

}  // namespace

/**
 * Synchronously displays the given web online page in the system WebApplet.
 */
app::OnlineWebExitData app::DisplayOnlineWebPage(const std::string& url) NN_NOEXCEPT {
  OnlineWebExitData result;

  nn::web::ShowWebPageArg showWebPageArg(url.c_str());
  showWebPageArg.SetMediaAutoPlayEnabled(true);

  showWebPageArg.SetFooterFixedKind(nn::web::WebFooterFixedKind_Auto);
  showWebPageArg.SetPointerEnabled(false);
  showWebPageArg.SetBootDisplayKind(nn::web::WebBootDisplayKind_Black);
  showWebPageArg.SetJsExtensionEnabled(true);
  showWebPageArg.SetTouchEnabledOnContents(true);
  showWebPageArg.SetPageScrollIndicatorEnabled(false);

  // Configure callback url
  showWebPageArg.SetCallbackUrl(kStarshotIpcCallbackUrlPrefix);

  nn::web::WebPageReturnValue webPageReturnValue;

  // Open the webview.  This is a blocking call.
  const nn::Result showOnlineResult = nn::web::ShowWebPage(&webPageReturnValue, showWebPageArg);
  
  if (showOnlineResult.IsSuccess()) {
    HandleResult(webPageReturnValue, result);
    NN_LOG("Exit Reason %d", result.reason);

    switch (result.reason) {
      case nn::web::WebExitReason_CallbackUrlReached:
        break;
      case nn::web::WebExitReason_EndButtonPressed:
        break;
      case nn::web::WebExitReason_BackButtonPressed:
        break;
      case nn::web::WebExitReason_ExitMessage:
        break;
      case nn::web::WebExitReason_LastWindowDeleted:
        break;
      case nn::web::WebExitReason_LocalNetworkDisconnectionDenied:
        break;
      case nn::web::WebExitReason_MediaPlayerClosed:
        break;
      case nn::web::WebExitReason_CausedByWebPage:
        break;
      case nn::web::WebExitReason_UserSelectionCanceled:
        break;
      case nn::web::WebExitReason_NetworkConnectionFailed:
        break;
      case nn::web::WebExitReason_SystemUpdateRequired:
        break;
      case nn::web::WebExitReason_CallerDataCorrupted:
        break;
      case nn::web::WebExitReason_Unexpected:
        break;
    }

  } else {
    NN_LOG("Cancel");

    result.reason = nn::web::WebExitReason::WebExitReason_Unexpected;
  }

  return result;
}

std::string app::GenerateStarshotUrl(
  const std::string& deviceId, const std::string& oauthToken, const std::string& refreshToken, bool chattingAllowed, const std::string& context) {
  std::string url(kStarshotUrl);

  // Add the route that will write params as cookies then redirect to the home page
  url += "/app-shell";

  url += "?unique_id=" + UrlEncode(deviceId);
  url += "&lnchv=" + UrlEncode(GetApplicationVersion());
  url += "&ipc-version=" + UrlEncode(kStarshotNativeInterfaceVersion);

  if (!chattingAllowed) {
    url += "&chat-allowed=false";
  }
  if (!context.empty()) {
    url += "&context=" + UrlEncode(context);
  }

  if (!oauthToken.empty()) {
    url += "#access-token=" + UrlEncode(oauthToken);

    if (!refreshToken.empty()) {
      url += "&refresh-token=" + UrlEncode(refreshToken);
    }
  }

  return url;
}
