﻿#include "app/save.hpp"
#include "app/utility.hpp"

#include "json11/json11.hpp"

#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <stdint.h>
#include <stdio.h>

#include <string>

namespace {
using namespace app;

constexpr char kMountPoint[] = "save";
constexpr char kFileName[] = "TwitchData";

constexpr size_t kSaveDataMaxSize = 1024 * 4;  // 4KB

std::string GetFilePath() {
  std::string result(kMountPoint);
  result += ":/";
  result += kFileName;
  return result;
}

void Mount(const nn::account::Uid& userId) {
  auto result = nn::fs::MountSaveData(kMountPoint, userId);
  NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void Unmount() {
  nn::fs::Unmount(kMountPoint);
}

constexpr char kOAuthTokenKey[] = "access-token";
constexpr char kRefreshTokenKey[] = "refresh-token";

void Serialize(const TwitchSaveData& data, char buffer[kSaveDataMaxSize]) {
  json11::Json::object root;
  root[kOAuthTokenKey] = data.oauthToken;
  root[kRefreshTokenKey] = data.refreshToken;
  auto serialized = json11::Json(root).dump();
  snprintf(buffer, kSaveDataMaxSize, "%s", serialized.data());
}

void Deserialize(const char buffer[kSaveDataMaxSize], TwitchSaveData& result) {
  result.oauthToken.clear();
  result.refreshToken.clear();

  std::string error;
  const auto root = json11::Json::parse(buffer, error, json11::JsonParse::STANDARD);
  if (error.empty()) {
    const auto& jsonOAuthToken = root[kOAuthTokenKey];
    if (jsonOAuthToken.is_string()) {
      result.oauthToken = jsonOAuthToken.string_value();
    }
    const auto& jsonRefreshToken = root[kRefreshTokenKey];
    if (jsonRefreshToken.is_string()) {
      result.refreshToken = jsonRefreshToken.string_value();
    }
  } else {
    NN_LOG("Error parsing save data json, ignoring");
  }
}

void WriteData(const std::string& path, const TwitchSaveData& data) {
  char buffer[kSaveDataMaxSize] = {0};
  Serialize(data, buffer);

  nn::fs::FileHandle fileHandle;

  auto result = nn::fs::OpenFile(&fileHandle, path.c_str(), nn::fs::OpenMode_Write);
  NN_ABORT_UNLESS_RESULT_SUCCESS(result);

  result = nn::fs::WriteFile(
    fileHandle, 0, buffer, kSaveDataMaxSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
  NN_ABORT_UNLESS_RESULT_SUCCESS(result);

  nn::fs::CloseFile(fileHandle);

  result = nn::fs::Commit(kMountPoint);
  NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void EnsureFileExists(const std::string& path, size_t size) {
  // See if the file exists yet
  nn::fs::DirectoryEntryType directoryEntryType;
  auto result = nn::fs::GetEntryType(&directoryEntryType, path.c_str());
  if (!nn::fs::ResultPathNotFound().Includes(result)) {
    return;
  }

  // Create the file
  result = nn::fs::CreateFile(path.c_str(), size);

  if (nn::fs::ResultPathNotFound::Includes(result)) {
    // The parent directory for the target file cannot be found.
    // This error handling is not necessary when parent directories exist without fail.
  } else if (nn::fs::ResultPathAlreadyExists::Includes(result) || nn::fs::ResultTargetLocked::Includes(result)) {
    // The target file already exists.
    // This error handling is not needed when the file already exists and that is not an issue.
    // Note that there is no assurance that the file size is equal to FileSize when you are not using error handling.
    // Delete the target file and re-create it when necessary.
  } else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result)) {
    // Insufficient data save region for the save data.
    NN_ABORT("Usable space not enough.");
  }
  // No further error handling is necessary because
  // the process is aborted in the library if it fails for any other reason.

  // Write default values
  TwitchSaveData data;
  WriteData(path, data);
}

}  // namespace

void app::SaveData(TwitchSaveData& data) {
  UseUserHandle([&data](nn::account::UserHandle userHandle, nn::account::Uid userId) {
    Mount(userId);

    auto filePath = GetFilePath();
    EnsureFileExists(filePath, kSaveDataMaxSize);

    WriteData(filePath, data);

    Unmount();
  });
}

void app::LoadData(TwitchSaveData& data) {
  data.oauthToken.clear();
  data.refreshToken.clear();

  UseUserHandle([&data](nn::account::UserHandle userHandle, nn::account::Uid userId) {
    Mount(userId);

    auto filePath = GetFilePath();

    nn::fs::FileHandle fileHandle;

    // Open the file
    auto result = nn::fs::OpenFile(&fileHandle, filePath.c_str(), nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
    if (result.IsSuccess()) {
      // Read the file
      char buffer[kSaveDataMaxSize] = {0};
      size_t readSize = 0;
      result = nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, kSaveDataMaxSize);
      NN_ABORT_UNLESS_RESULT_SUCCESS(result);

      Deserialize(buffer, data);

      nn::fs::CloseFile(fileHandle);
    } else {
      // Not finding the file is ok
      if (!nn::fs::ResultPathNotFound::Includes(result)) {
        NN_ABORT("LoadData: failed to open file");
      }
    }

    Unmount();
  });
}
