﻿#include "app/graphics.hpp"
#include "app/splash.hpp"
#include "app/utility.hpp"

#include <EGL/egl.h>
#include <EGL/eglext.h>
#include <GL/glew.h>
#include <nn/gfx.h>
#include <nn/image/image_PngDecoder.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/vi.h>
#include <nv/nv_MemoryManagement.h>

#include <algorithm>
#include <cstdlib>
#include <vector>

namespace {
using namespace app;

// clang-format off
// 
// A vertex shader that passes through coordinates that are already
// in screen space
constexpr GLchar kVertexShaderSource[] =
  "#version 420\n"
  "precision highp float;\n"
  "const vec2 position[6] = vec2[]( vec2(-1, 1), vec2(1, 1), vec2(-1, -1), vec2(-1, -1), vec2(1, 1), vec2(1, -1) );\n"
  "const vec2 texcoord[6] = vec2[]( vec2(0, 0),  vec2(1, 0), vec2(0, 1),   vec2(0, 1),   vec2(1, 0), vec2(1, 1) );\n"
  "out vec2 TexCoord;\n"
  "void main() {\n"
  "  gl_Position = vec4(position[gl_VertexID], 0.0, 1.0);\n"
  "  TexCoord = texcoord[gl_VertexID];\n"
  "}\n";

constexpr GLchar kFragmentShaderSource[] =
  "#version 420\n"
  "precision highp float;\n"
  "layout (location = 0) out vec4 o_Color;\n"
  "in vec2 TexCoord;\n"
  "layout (binding = 0) uniform sampler2D u_Texture;\n"
  "void main() {\n"
  "  o_Color = texture(u_Texture, TexCoord);\n"
  "}\n";

constexpr GLuint kNumTriangles = 2;

GLuint gVertexShaderId = 0;
GLuint gFragmentShaderId = 0;
GLuint gShaderProgramId = 0;
GLuint gVertexArrayObjectId = 0;
GLuint gTextureUniformIndex = 0;
GLuint gTextureId = 0;

// clang-format on

#pragma region Memory Allocation

void* CustomAllocate(size_t size, void* /*pUserData*/) {
  return malloc(size);
}

void CustomFree(void* p, void* /*pUserData*/) {
  free(p);
}

#pragma endregion

bool DecodePngData(std::vector<uint8_t>& outputBuffer, nn::image::Dimension& outputDimensions,
  const std::vector<uint8_t>& encodedData) NN_NOEXCEPT {

  nn::image::PngDecoder decoder;
  decoder.Initialize(CustomAllocate, nullptr, CustomFree, nullptr);
  decoder.SetImageData(encodedData.data(), encodedData.size());
  decoder.SetOutputPixelFormat(nn::image::PixelFormat::PixelFormat_Rgba32);

  // Make sure it's a valid PNG
  nn::image::PngStatus status = decoder.Analyze();
  if (status != nn::image::PngStatus_Ok) {
    NN_LOG("decoder.Analyze() failed");
    return false;
  }

  // Allocate the output buffer
  outputDimensions = decoder.GetDimension();
  size_t rowStride = decoder.GetRowStrideMin();
  size_t pixelDataSize = rowStride * outputDimensions.height;
  outputBuffer.resize(pixelDataSize);

  // Decode
  status = decoder.Decode(outputBuffer.data(), pixelDataSize, rowStride);
  return status == nn::image::PngStatus_Ok;
}

bool LoadPngFileIntoTexture(const std::string& path) {
  // Load the PNG file from assets
  std::vector<uint8_t> encodedPng;
  bool succeeded = LoadFile(path, encodedPng);
  if (!succeeded) {
    return false;
  }

  // Decode the PNG
  std::vector<uint8_t> decodedPng;
  nn::image::Dimension pngImageDimension;
  if (!DecodePngData(decodedPng, pngImageDimension, encodedPng)) {
    NN_LOG("Failed to decode PNG data");
    return false;
  }

  // Create texture from buffer
  glGenTextures(1, &gTextureId);
  glBindTexture(GL_TEXTURE_2D, gTextureId);
  glTexStorage2D(GL_TEXTURE_2D, 0, GL_RGB, pngImageDimension.width, pngImageDimension.height);
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, pngImageDimension.width, pngImageDimension.height, 0, GL_RGBA,
    GL_UNSIGNED_BYTE,
    decodedPng.data());
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

  return true;
}

void InitializeShaders() {
  GLint result;
  GLchar shaderLog[256];
  GLsizei shaderLogSize;
  const GLchar* stringArr;

  // Create and compile the vertex shader
  gVertexShaderId = glCreateShader(GL_VERTEX_SHADER);
  NN_ASSERT(gVertexShaderId != 0, "Failed to create vertex shader\n");
  stringArr = &kVertexShaderSource[0];
  glShaderSource(gVertexShaderId, 1, &stringArr, nullptr);
  glCompileShader(gVertexShaderId);
  glGetShaderiv(gVertexShaderId, GL_COMPILE_STATUS, &result);
  if (result == 0) {
    glGetShaderInfoLog(gVertexShaderId, sizeof(shaderLog), &shaderLogSize, shaderLog);
    NN_ASSERT(false, "Failed to compile vertex shader: %s\n", shaderLog);
  }

  // Create and compile the fragment shader
  gFragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
  NN_ASSERT(gFragmentShaderId != 0, "Failed to create fragment shader\n");
  stringArr = &kFragmentShaderSource[0];
  glShaderSource(gFragmentShaderId, 1, &stringArr, nullptr);
  glCompileShader(gFragmentShaderId);
  glGetShaderiv(gFragmentShaderId, GL_COMPILE_STATUS, &result);
  if (result == 0) {
    glGetShaderInfoLog(gFragmentShaderId, sizeof(shaderLog), &shaderLogSize, shaderLog);
    NN_ASSERT(false, "Failed to compile fragment shader: %s\n", shaderLog);
  }

  // Link the final program
  gShaderProgramId = glCreateProgram();
  NN_ASSERT(gShaderProgramId != 0, "Failed to create shader program\n");
  glAttachShader(gShaderProgramId, gVertexShaderId);
  glAttachShader(gShaderProgramId, gFragmentShaderId);
  glLinkProgram(gShaderProgramId);
  glGetProgramiv(gShaderProgramId, GL_LINK_STATUS, &result);
  if (result == 0) {
    glGetProgramInfoLog(gShaderProgramId, sizeof(shaderLog), &shaderLogSize, shaderLog);
    NN_ASSERT(false, "Failed to link shader program: %s\n", shaderLog);
  }

  // Extract attributes and uniforms
  gTextureUniformIndex = glGetUniformLocation(gShaderProgramId, "u_Texture");
}

void InitializeVertexData() {
  // We required an empty vertex buffer object
  glGenVertexArrays(1, &gVertexArrayObjectId);
}

}  // namespace

void InitializeTwitchSplashScreen() {
  InitializeShaders();
  InitializeVertexData();
  LoadPngFileIntoTexture("Asset:/Splash.png");
}

void ShowTwitchSplashScreen() {
  // Clear the screen
  glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
  glClear(GL_COLOR_BUFFER_BIT);

  // Render the fullscreen quad
  glUseProgram(gShaderProgramId);
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, gTextureId);
  glBindVertexArray(gVertexArrayObjectId);
  glDrawArrays(GL_TRIANGLES, 0, kNumTriangles * 3);

  // Display the result of rendering
  SwapBuffers();
}

void ShutdownTwitchSplashScreen() {
  glDetachShader(gShaderProgramId, gVertexShaderId);
  glDetachShader(gShaderProgramId, gFragmentShaderId);
  glDeleteShader(gVertexShaderId);
  glDeleteShader(gFragmentShaderId);
  glDeleteProgram(gShaderProgramId);
  glDeleteVertexArrays(1, &gVertexArrayObjectId);
  glDeleteTextures(1, &gTextureId);
}
