﻿#include "app/utility.hpp"
#include "core/time.hpp"

#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/oe.h>
#include <nn/time/time_SystemClockTraits.h>
#include <nn/time/time_StandardUserSystemClock.h>

#include <time.h>

namespace {
void OpenUser(nn::account::UserHandle& userHandle, nn::account::Uid& userId) {
  userId = nn::account::InvalidUid;

  // Open the user selected at application startup.
  // Keep it open for the duration of user operations.
  bool userSelected = nn::account::TryOpenPreselectedUser(&userHandle);

  if (!userSelected) {
    NN_ABORT("Failed to select system user id");
  }

  // Get the user identifier that was selected at application startup.
  auto result = nn::account::GetUserId(&userId, userHandle);
  NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}
}

std::string app::GetDeviceId() {
  nn::util::Uuid uuid;
  char buffer[nn::util::Uuid::StringSize];

  // GetPseudoDeviceId returns a stable device id that is unique for the app/device combination
  nn::oe::GetPseudoDeviceId(&uuid);
  uuid.ToString(buffer, nn::util::Uuid::StringSize);
  std::string result = buffer;

  // Remove '-' characters
  result.erase(remove(result.begin(), result.end(), '-'), result.end());

  return result;
}

std::string app::GetApplicationVersion() {
  nn::oe::DisplayVersion displayVersion;
  nn::oe::GetDisplayVersion(&displayVersion);
  return displayVersion.value;
}

// This is declared in the common library but defined here in the Switch-specific project.
uint64_t GetSecondsSinceEpoch() {
  nn::time::SystemClockTraits::time_point tp = nn::time::StandardUserSystemClock::now();
  return static_cast<uint64_t>(nn::time::StandardUserSystemClock::to_time_t(tp));
}

bool app::LoadFile(const std::string& path, std::vector<uint8_t>& result) {
  // Open the file
  nn::fs::FileHandle fileHandle;
  auto ret = nn::fs::OpenFile(&fileHandle, path.c_str(), nn::fs::OpenMode_Read);
  if (!ret.IsSuccess()) {
    NN_LOG("Failed to open file \"%s\"", path.c_str());
    return false;
  }

  bool succeeded = true;

  // Determine file size
  int64_t fileSize = 0;
  ret = nn::fs::GetFileSize(&fileSize, fileHandle);
  if (!ret.IsSuccess()) {
    NN_LOG("Failed to get file size \"%s\"", path.c_str());
    succeeded = false;
  }

  // Load the file contents
  if (succeeded) {
    result.resize(static_cast<size_t>(fileSize));
    ret = nn::fs::ReadFile(fileHandle, 0, result.data(), result.size());
    if (!ret.IsSuccess()) {
      NN_LOG("Failed to read file \"%s\"", path.c_str());
      succeeded = false;
    }
  }

  nn::fs::CloseFile(fileHandle);

  return succeeded;
}

void app::UseUserHandle(const std::function<void(nn::account::UserHandle userHandle, nn::account::Uid userId)>& func) {
  nn::account::UserHandle userHandle = {{0}};
  nn::account::Uid userId = nn::account::InvalidUid;
  OpenUser(userHandle, userId);

  func(userHandle, userId);

  CloseUser(userHandle);
}
