#pragma once

#include "app/playstationplatformuser.h"
#include "app/utils.h"
#include "json11/json11.hpp"

#include <save_data.h>
#include <save_data_dialog.h>
#include <scebase.h>
#include <user_service/user_service_defs.h>

#include <functional>
#include <memory>
#include <string>

namespace app {
namespace sce {
class StorageManager;
}  // namespace sce
}  // namespace app

/**
  * PlaystationStorageManager provides common StorageManager functionality for playstation platforms. Playstation platforms provide concrete implementations of this class.
  */
class app::sce::StorageManager {
 public:
  StorageManager();
  ~StorageManager();
  int32_t Initialize();
  int32_t Terminate();

  // virtual void PostInitialize();

  // virtual ErrorPtr BeginAccesses(const std::shared_ptr<PlatformUser>& user);
  // virtual ErrorPtr EndAccesses(const std::shared_ptr<PlatformUser>& user);

  bool SetDeviceData(const std::string& containerName, const std::string& key, const json11::Json::object& data);
  bool GetDeviceData(const std::string& containerName, const std::string& key, json11::Json result);
  bool SetUserData(const std::shared_ptr<PlaystationPlatformUser>& user, const std::string& container,
    const std::string& key, const json11::Json& data);
  bool GetUserData(const std::shared_ptr<PlaystationPlatformUser>& user, const std::string& container,
    const std::string& key, json11::Json result);

  bool HandleCorruptData(SceUserServiceUserId userId);
  // AsyncResultPtr<void> HandleNoSpace( const std::shared_ptr<PlaystationPlatformUser>& user, const std::string& containerName, uint64_t numBlocks);

 protected:
  struct MountParams {
    SceUserServiceUserId userId = 0;
    std::string containerName;
    SceSaveDataMountMode mountMode = 0;
    SceSaveDataBlocks numBlocks = 0;
    int32_t transactionResourceId = 0;
    bool allowRecreationAttempt = false;
  };

  /**
   * Call a platform specific mount function and return the error code if any. This needs to be called before interacting with user save data.
   */
  virtual int32_t SceMount(const MountParams& mountParams, SceSaveDataMountResult& result) = 0;

  /**
   * Call a platform specific unmount function and return the error code if any. This needs to be called after all interaction with user save data is complete.
   */
  virtual int32_t SceUnmount(const SceSaveDataMountPoint& mountPoint) = 0;

  /**
   * On Prospero the concept of a transaction resource id was introduced, a valid id needs to be passed to all mount/unmount functions. On platforms other than Prospero this returns 0.
   */
  virtual int32_t GetTransactionResourceId() = 0;

  /**
   * Before writing user save data this function must be called
   */
  virtual int32_t ScePrepare(const SceSaveDataMountPoint& mountPoint) = 0;

  /**
   * After writing user save data this function must be called
   */
  virtual int32_t SceCommit() = 0;

  /**
   * Subclasses can override to provide definitions for platform specific error codes
   */
  virtual bool CheckErrorCode(int32_t ec);

 private:
  // ErrorPtr HandleNoSpaceError(SceUserServiceUserId userId, const std::string& containerName, uint64_t numBlocks);
  bool SafeMount(const MountParams& params, const std::function<bool(const SceSaveDataMountResult&)>& callback);
  bool Mount(const MountParams& params, SceSaveDataMountResult& result);
  bool Unmount(const SceSaveDataMountResult& mount);
  bool DeleteContainer(SceUserServiceUserId userId, const std::string& containerName);
  bool WriteFileWithTransaction(const std::string& directory, const std::string& fileName, const std::string& data);
  bool LoadFile(const std::string& directory, const std::string& fileName, std::string& result);

  /**
   * Retrieves the TaskRunner allocated to the given container.  If none has been allocated yet it will be created.
   */
  // std::shared_ptr<ttv::TaskRunner> GetTaskRunnerForContainer(const std::string& container);

  /**
   * The task runners for each independent container.  The key is the container being processed.  This serializes the accesses for a
   * particular container.
   */
  // std::map<std::string, std::shared_ptr<ttv::TaskRunner>> mTaskRunners;

  /**
   * The task runner used for reading and writing blob files.
   */
  // std::shared_ptr<ttv::TaskRunner> mBlobTaskRunner;

  /**
   * The mutex used to manage mTaskRunners.
   */
  // std::unique_ptr<ttv::IMutex> mTaskRunnerMutex;

  /**
   * The number of calls to BeginAccesses() which have not been ended.
   */
  uint32_t mNumBeginAccesses;
  /**
   * The user that currently has a locked mount on the shared container.
   */
  std::shared_ptr<PlaystationPlatformUser> mSharedContainerMountedUser;

  /**
   * The data for the shared container if it is currently mounted.  This is accessed from the background thread.
   */
  SceSaveDataMountResult mSharedContainerMountData;
};
