#!/usr/bin/python
#
# This script configures the current PS4 build for a specific region.
#
# At minimum you must specify the following
#
#   configure_build.py --region=siea --volume-type=remaster --twitch-build-number=12345 --commit-hash=ae34asdf23cv34 --sce-master-version=01.00 --sce-app-version=01.17 --sce-title-id=CUSA03285 --sce-content-id=UP8902-CUSA03285_00-TWITCHPS4APPSCEA
#

# Import standard modules
import argparse
import imp
import json
import os
import os.path
import shutil
import subprocess
import sys
import time

# Load other scripts for use
script_dir = os.path.realpath(os.path.dirname(__file__))
ps4_tools = imp.load_source('ps4_tools', os.path.join(script_dir, 'ps4_tools.py'))

tools_dir = script_dir#os.path.realpath(os.path.join(script_dir, '..', '..', 'tools'))
configure_version = imp.load_source('configure_version', os.path.join(tools_dir, 'configure_version.py'))

class ConfigureOptions:
    def __init__(self):
        self.package_dir = os.path.realpath(os.path.join(script_dir, '..'))
        self.meta_dir = os.path.realpath(os.path.join(script_dir, '..', 'meta'))
        self.temp_dir = os.path.realpath(os.path.join(script_dir, '..', 'layout_temp'))
        self.platform = 'ps4'
        self.region = None
        #self.twitch_version = configure_version.get_version(self.platform) # set in vesions.json
        self.twitch_build_number = None
        self.commit_hash = None
        self.volume_type = 'remaster' # 'remaster' or 'patch'
        self.sce_master_version = None
        self.sce_app_version = None
        self.sce_title_id = None
        self.sce_content_id = None
        self.sce_nptitle_dat = None
        self.use_dummy_nptitledat = False


def parse_command_line(explicit_arguments, config):

    """Parses the command line."""

    parser = argparse.ArgumentParser(description='Configures the Twitch PS4 app.')

    parser.add_argument(
        '--twitch-build-number',
        required=True,
        metavar='<twitch_build_number>',
        help="Specifies the Twitch build number."
    )

    parser.add_argument(
        '--commit-hash',
        required=True,
        metavar='<commit_hash>',
        help="Specifies the Git commit hash."
    )

    parser.add_argument(
        '--volume-type',
        required=False,
        metavar='<volume-type>',
        help="Specifies the type of package.  Options are 'remaster' or 'patch'.  Defaults to 'patch'"
    )

    parser.add_argument(
        '--sce-master-version',
        required=True,
        metavar='<sce_master_version>',
        help="Specifies the VERSION attribute to be written into the param.sfo file"
    )

    parser.add_argument(
        '--sce-app-version',
        required=True,
        metavar='<sce_app_version>',
        help="Specifies the APP_VER attribute to be written into the param.sfo file"
    )

    parser.add_argument(
        '--sce-title-id',
        required=False,
        metavar='<sce_title_id>',
        help="Specifies the title id to be written into the param.sfo file"
    )

    parser.add_argument(
        '--sce-content-id',
        required=False,
        metavar='<sce_content_id>',
        help="Specifies the content id"
    )

    parser.add_argument(
        '--use-dummy-nptitledat',
        required=False,
        action='store_true',
        help='Specify to use a dummy title ID for building the app. This can be used if you do not have access to the internal Twitch network'
    )

    ps4_tools.setup_regions_argument(parser)

    args = parser.parse_args(explicit_arguments)

    if args.region:
        config.region = args.region.lower()

    if args.twitch_build_number:
        config.twitch_build_number = args.twitch_build_number

    if args.commit_hash:
        config.commit_hash = args.commit_hash

    if args.volume_type:
        config.volume_type = args.volume_type

    if args.sce_master_version:
        config.sce_master_version = args.sce_master_version

    if args.sce_app_version:
        config.sce_app_version = args.sce_app_version

    if args.sce_title_id:
        config.sce_title_id = args.sce_title_id

    if args.sce_content_id:
        config.sce_content_id = args.sce_content_id

    if args.use_dummy_nptitledat:
        config.use_dummy_nptitledat

    # Get the default values for region-specific params
    title_info = ps4_tools.get_region_title_info(config.region, config.use_dummy_nptitledat)

    if config.sce_title_id is None:
        config.sce_title_id = title_info.sce_title_id

    if config.sce_content_id is None:
        config.sce_content_id = title_info.sce_content_id

    config.sce_nptitle_dat = title_info.sce_nptitle_dat


def load_file(path):

    with open(path, 'r') as file:
        return file.read().decode('utf-8')


def save_file(path, data):

    with open(path, 'w') as file:
        file.write(data)


def configure_param_sfo(config):

    print("Configuring param.sfo...")

    sony_tools_config = ps4_tools.find_sony_tools()

    # Setup the param.sfo file for the region
    sfx_file = ps4_tools.SfxFile(sony_tools_config)

    sfx_template_path = os.path.join(config.package_dir, 'config', 'template', 'param.sfx')
    sfx_file.load_sfx(sfx_template_path)

    sfx_file.set_master_version(config.sce_master_version)
    sfx_file.set_app_version(config.sce_app_version)
    sfx_file.set_title_id(config.sce_title_id)
    sfx_file.set_content_id(config.sce_content_id)
    sfx_file.set_remote_play_key_assign("1") # This is what was in the existing sfo files from Loot

    if config.region == 'siea':
        sfx_file.set_user_defined_param(1, 0)
    elif config.region == 'siee':
        sfx_file.set_user_defined_param(1, 1)
    elif config.region == 'sieasia':
        sfx_file.set_user_defined_param(1, 2)
    else:
        raise Exception('Unhandled region: ' + config.region)

    if config.volume_type == 'remaster':
        sfx_file.set_category('gde')
        sfx_file.set_remaster_type('1')
    elif config.volume_type == 'patch':
        sfx_file.set_category('gpe')
    else:
        raise Exception('Unhandled volume_type: ' + config.volume_type)

    # Set Parental Lock Level. SIEE = 9, SIEAsia = 5, SIEA = 5
    if config.region == 'siee':
        sfx_file.set_parental_level('9')
    else:
        sfx_file.set_parental_level('5')

    # Constants
    neo_attribute = 8388608
    circle_attribute = 0
    cross_attribute = 2
    assigned_by_system_attribute = 32

    attributes = assigned_by_system_attribute

    sfx_file.set_attributes(str(attributes))

    sfo_path = os.path.join(config.package_dir, 'config', config.region, 'sce_sys', 'param.sfo')
    sfx_file.save_sfo(sfo_path)

    print("Done configuring param.sfo.")

    print("Copying package data from config folder")

    if os.path.exists(os.path.join(config.temp_dir, 'sce_sys')):
        shutil.rmtree(os.path.join(config.temp_dir, 'sce_sys'))

    shutil.copytree(os.path.join(config.package_dir, 'config', config.region, 'sce_sys'), os.path.join(config.temp_dir, 'sce_sys'))

    print("Done copying package data from config folder")


def setup_environment_variables(config):

    settings_path = os.path.join(config.meta_dir, 'settings.json')
    if not os.path.isfile(settings_path):
        raise Exception('PS4 settings.json file not found')

    with open(settings_path) as json_file:
        settings = json.load(json_file)

    ps4_sdk_path = 'C:\\Program Files (x86)\\SCE\\ORBIS SDKs\\' + settings['PS4_SDK']

    os.environ['SCE_ORBIS_SDK_DIR'] = ps4_sdk_path

    print('Setting environment variable SCE_ORBIS_SDK_DIR = ' + ps4_sdk_path)

# Download the nptitle.dat from Sandstorm
def configure_nptitle(config):
    nptitle_path = os.path.join(config.package_dir, 'config', config.region, 'sce_sys', 'nptitle.dat')
    with open(nptitle_path, 'wb') as f:
        print('Writing nptitle.dat for ' + config.region + ' at ' + nptitle_path)
        f.write(config.sce_nptitle_dat)

def configure_all(config):
    setup_environment_variables(config)
    configure_nptitle(config)
    configure_param_sfo(config)


# Process the command line arguments if run as the primary script
if __name__ == "__main__":
    # Parse the command line
    config = ConfigureOptions()
    parse_command_line(None, config)

    configure_all(config)
