#!/usr/bin/python
#
# Helper function to fetch tvapps-prod secrets from Sandstorm
#
# Usage example:
#
#   mySecret = keys.getSecret('playstation', 'platforms/PlayStation/production/some_secret')
#
#
# Make sure you have sandstorm installed, and that you have your credentials set up in ~/.aws/credentials
# For more info: https://wiki.twitch.com/display/ENG/Setting+Up+Sandstorm
#

import subprocess
import os

profile = 'tvapps-prod'
roles = {
    'playstation': 'arn:aws:iam::734326455073:role/sandstorm/production/templated/role/playstation',
    'android':     'arn:aws:iam::734326455073:role/sandstorm/production/templated/role/android',
    'build':       'arn:aws:iam::734326455073:role/sandstorm/production/templated/role/build',
    'xbox':        'arn:aws:iam::734326455073:role/sandstorm/production/templated/role/xbox'
}

def getSecret(role, secret):
    """Get a secret string from Sandstorm
    role -- The role for the secret, for example 'build'
    secret -- The secret name, for example 'platforms/build/production/youi_apikey'
    """
    if role not in roles:
        raise Exception('Role "' + role + '" not defined')
    try:
        # Sandstorm arguments have to be in a specific order
        # Only add -profile arg if SANDSTORM_NO_CREDENTIALS_PROFILE env variable isn't present
        scriptName = ['sandstorm', 'sandstorm.exe'][os.name == 'nt']
        shell_args = scriptName +  ' get'
        if 'SANDSTORM_NO_CREDENTIALS_PROFILE' not in os.environ:
            shell_args += ' -profile ' + profile
        shell_args += ' -r ' + roles[role] + ' -n ' + secret

        proc = subprocess.Popen(shell_args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
        proc.wait()
        out, err = proc.communicate()
    except Exception as e:
        raise Exception("Make sure sandstorm is installed", e)
    else:
        if proc.returncode != 0:
            print(err)
            print("Make you have credentials for " + profile + " defined in ~/.aws/credentials")
            raise Exception('Get secret failed')
        return out

def fetchKeystore(key, save_path):
    """Fetch an Android keystore file and password, store them in save_path
    key --- The key to fetch, "release" or "upload"
    save_path --- Path where the keystore and password files will be saved

    This pulls the PEM file from Sandstorm, along with the cert password.
    The PEM has to be converted into a PKCS12 file, using the cert password
    """
    print('Getting %s keystore'%key)

    temp_file_path = '{}/{}.p12'.format(save_path, key)
    keystore_path = '{}/{}.keystore'.format(save_path, key)
    password_path = '{}/{}.keystore.password'.format(save_path, key)

    alias = 'twitch-%s'%key
    role = 'android'

    # Remove existing keystore and password files
    for path in [keystore_path, password_path]:
        if os.path.exists(path):
          os.remove(path)

    # Pull the keystore from Sandstorm
    pem = getSecret(role, 'platforms/Android/production/{}keystore'.format(key))

    # Pull the passwork from Sandstorm
    password = getSecret(role, 'platforms/Android/production/certpassword')

    # Save the password to file
    with open(password_path, 'w') as password_file:
        password_file.write(password)

    # Save password as environment variable
    os.environ['KEYTOOL_PASSWORD'] = password

    # Create p12 file from pem
    command = 'echo "{input}" | openssl pkcs12 -export -out {output_path} -passin pass:"{password}" -passout pass:"{password}" -name "{alias}"'.format(input=pem, output_path=temp_file_path, password=password, alias=alias)
    subprocess.call(command, shell=True)

    # Create keystore
    command = 'keytool -importkeystore -deststorepass "{password}" -destkeystore "{keystore_path}" -deststoretype PKCS12 -srckeystore "{output_path}" -srcstoretype PKCS12 -srcstorepass "{password}"'.format(password=password, output_path=temp_file_path, keystore_path=keystore_path)
    subprocess.call(command, shell=True)

    # Delete temp file
    os.remove(temp_file_path)

    # List the certificate info
    # command = 'keytool -list -v -keystore "{keystore_path}" -alias "{alias}" -storepass "{password}"'.format(keystore_path=keystore_path, alias=alias, password=password)
    # subprocess.call(command, shell=True)

    print('Keystore file saved to %s'%os.path.realpath(keystore_path))
