#!/usr/bin/python
#
# This script facilitates PS4 package generation
#
# At minimum you must specify the following
#
#   package.py --build-config=Release --region=siea --volume-type=remaster --sce-app-version=01.20
#

# Import standard modules
import sys
import os
import argparse
import shutil
import subprocess
import time
import imp
import glob
import xml.etree.ElementTree
import re
import uuid
import boto3
import zipfile

# Load other scripts for use
scripts_dir = os.path.realpath(os.path.dirname(__file__))
ps4_tools = imp.load_source('ps4_tools', os.path.join(scripts_dir, 'ps4_tools.py'))
package_dir = os.path.join(scripts_dir, '..')
meta_dir = os.path.join(package_dir, 'meta')
ps4_patchnotes_dir = os.path.join(meta_dir, 'patch_notes')
layout_dir = os.path.join(package_dir, 'layout_temp')

class Configuration:

    def __init__(self):
        self.output_path = os.path.realpath(os.path.join(package_dir, 'submission')) # The output directory
        self.package_prefix = os.path.realpath(os.path.join(package_dir, 'releases')) # The previous package directory
        self.app_name = 'StarshotPS4'
        self.region = None
        self.build_config = 'release'
        self.master_package_path = None # The original release package of the app
        self.latest_package_path = None # The latest release package of the app
        self.sce_app_version = None
        self.sce_title_id = None
        self.sce_content_id = None
        self.sce_passcode = None
        self.use_dummy_nptitledat = False
        self.sce_tools_info = None
        self.sfx_file = None # The instance of ps4_tools.SfxFile that holds the param.sfo data
        self.staging_path = None
        self.volume_type = 'patch' # 'remaster' or 'patch'

        self.boto_s3_client = None

    def s3_client(self):
        if self.boto_s3_client:
            return self.boto_s3_client

        if 'USE_AWS_INSTANCE_PROFILE' not in os.environ:
            session = boto3.session.Session(profile_name='tvapps-prod')
        else:
            sts = boto3.client("sts")
            role_arn = "arn:aws:iam::969432690216:role/csi-teamcity"
            token = sts.assume_role(RoleArn=role_arn, RoleSessionName="teamcity-{}".format(str(uuid.uuid4())))
            credentials = token["Credentials"]
            session =  boto3.Session(aws_access_key_id=credentials["AccessKeyId"],
                                     aws_secret_access_key=credentials["SecretAccessKey"],
                                     aws_session_token=credentials["SessionToken"])

        self.boto_s3_client = session.client('s3')
        return self.boto_s3_client


def parse_command_line(explicit_arguments, config):

    """Parses the command line."""

    parser = argparse.ArgumentParser(description='Packages the Twitch PS4 app.')

    parser.add_argument(
        '--output-path',
        required=False,
        metavar='<output-path>',
        help="Specifies the directory to output the build"
    )

    parser.add_argument(
        '--build-config',
        required=True,
        metavar='<debug|release>',
        help="Specifies whether to package the 'debug' or 'release' build."
    )

    parser.add_argument(
        '--master-package-path',
        required=False,
        metavar='<master-pkg-path>',
        help="Specifies the path to the master package."
    )

    parser.add_argument(
        '--latest-package-path',
        required=False,
        metavar='<master-pkg-path>',
        help="Specifies the path to the latest release package."
    )

    parser.add_argument(
        '--volume-type',
        required=False,
        metavar='<volume-type>',
        help="Specifies the type of package.  Options are 'remaster' or 'patch'.  Defaults to 'patch'"
    )

    parser.add_argument(
        '--app-name',
        required=False,
        metavar='<app-name>',
        help="Specifies the name of the app to package."
    )

    parser.add_argument(
        '--sce-app-version',
        required=True,
        metavar='<sce_app_version>',
        help="Specifies the application version"
    )

    parser.add_argument(
        '--sce-title-id',
        required=False,
        metavar='<sce_title_id>',
        help="Specifies the title id"
    )

    parser.add_argument(
        '--sce-content-id',
        required=False,
        metavar='<sce_content_id>',
        help="Specifies the content id"
    )

    parser.add_argument(
        '--sce-passcode',
        required=False,
        metavar='<sce_passcode>',
        help="Specifies the passcode"
    )

    parser.add_argument(
        '--use-dummy-nptitledat',
        required=False,
        action='store_true',
        help='Specify to use a dummy title ID for building the app. This can be used if you do not have access to the internal Twitch network'
    )

    ps4_tools.setup_regions_argument(parser)

    args = parser.parse_args(explicit_arguments)

    if args.output_path:
        config.output_path = os.path.realpath(args.output_path)

    if args.build_config:
        config.build_config = args.build_config

    if args.region:
        config.region = args.region.lower()

    if args.master_package_path:
        config.master_package_path = args.master_package_path

    if args.latest_package_path:
        config.latest_package_path = args.latest_package_path

    if args.volume_type:
        config.volume_type = args.volume_type

    if args.app_name:
        config.app_name = args.app_name

    if args.sce_app_version:
        config.sce_app_version = args.sce_app_version

    if args.sce_title_id:
        config.sce_title_id = args.sce_title_id

    if args.sce_content_id:
        config.sce_content_id = args.sce_content_id

    if args.sce_passcode:
        config.sce_passcode = args.sce_passcode

    if args.use_dummy_nptitledat:
        config.use_dummy_nptitledat = args.use_dummy_nptitledat

    # Get the default values for region-specific params
    title_info = ps4_tools.get_region_title_info(config.region, config.use_dummy_nptitledat)

    if config.sce_title_id is None:
        config.sce_title_id = title_info.sce_title_id

    if config.sce_content_id is None:
        config.sce_content_id = title_info.sce_content_id

    if config.sce_passcode is None:
        config.sce_passcode = title_info.sce_passcode


def find_highest_version_package_in_list(file_list):
    files = list(filter(lambda x: x['Key'].endswith('.pkg'), file_list))
    if len(files) == 0:
        return None

    if (len(files) == 1):
        return (files[0]['Key'], files[0]['Size'])

    # Get the latest MDT if there are more than one file
    # For example, you may have: siea/remaster_01.44/UP8902-CUSA03285_00-TWITCHPS4APPSCEA-A0144-V0100.pkg
    # and                        siea/remaster_01.44/UP8902-CUSA03285_00-TWITCHPS4APPSCEA-A0144-V0101.pkg
    # Then you want to only download the V0101.pkg

    bestVersion = -1
    bestFile = ''
    bestSize = 0
    for f in files:
        file = f['Key']
        m = re.match(r'-V(\d+)\.pkg$', file)
        if (m != None) :
            version = float(m.group(0))
            if (version > bestVersion):
                bestVersion = version
                bestFile = file
                bestSize = f['Size']

    if (bestFile == ''):
        raise Exception("Couldn't find a single valid candidate package")

    if (bestSize == 0):
        raise Exception('Candidate package: ' + bestFile + ' has no size')

    return (bestFile, bestSize)


def clean(config):

    # Delete previous packages
    for path in os.listdir(package_dir):
        full_path = os.path.join(package_dir, path)
        if path.endswith(".pkg") and os.path.isfile(full_path):
            os.remove(full_path)

    # Clean and prepare the staging directory
    if os.path.isdir(config.staging_path):
        shutil.rmtree(config.staging_path)

        # Wait for the filesystem to settle
        time.sleep(5)


def verify_change_info(config):

    print('Verifying changeinfo.xml files: ' + config.sce_app_version)

    changeinfo_path = os.path.join(package_dir, 'config', config.region, 'sce_sys', 'changeinfo')

    # Get all XML files in the sce_sys/changeinfo directory
    files = os.listdir(changeinfo_path)

    for path in files:

        full_path = os.path.join(changeinfo_path, path)
        if not os.path.isfile(full_path) or not full_path.endswith('.xml'):
            continue

        print('Processing ' + full_path)

        # Load the XML doc
        doc = xml.etree.ElementTree.parse(full_path)
        if doc is None:
            raise Exception('Could not load XML doc ' + full_path)

        xml_changeinfo = doc.getroot()
        assert xml_changeinfo.tag == 'changeinfo'

        # Make sure we added an entry for the update we're making
        found_current_version = False
        for xml_changes in xml_changeinfo.findall("changes"):
            if not 'app_ver' in xml_changes.attrib:
                raise Exception('Could not find app_ver attribute')

            if xml_changes.attrib['app_ver'] == config.sce_app_version:
                found_current_version = True
                break

        if not found_current_version:
            raise Exception('Could not entry for current release in ' + full_path + '.  Make sure you update the patch notes in all the changeinfo.xml files.')

    print('Done verifying changeinfo.xml files.')


def stage_package(config):

    print('Staging package...')

    playstation_dir = os.path.join(package_dir, '..')

    os.makedirs(config.staging_path)

    sce_module_dir = os.path.join(config.staging_path, 'sce_module')
    os.makedirs(sce_module_dir)

    build_config_dir = os.path.join(package_dir, config.build_config)

    # Bring elf and map into the package dir for verification (unneeded, probably remove later)
    if not os.path.exists(build_config_dir):
        os.makedirs(build_config_dir)
    shutil.copy(os.path.join(playstation_dir, 'ORBIS_Debug', config.app_name + '.elf'), build_config_dir)
    shutil.copy(os.path.join(playstation_dir, 'ORBIS_Debug', config.app_name + '.map'), build_config_dir)

    # Package the .elf
    shutil.copy(os.path.join(playstation_dir, 'ORBIS_Debug', config.app_name + '.elf'), os.path.join(config.staging_path, 'eboot.bin'))

    # Package the assets
    # shutil.copytree(os.path.join(package_dir, 'assets'), os.path.join(config.staging_path, 'assets'))

    # Package config data
    shutil.copytree(os.path.join(package_dir, 'config', config.region, 'sce_sys'), os.path.join(config.staging_path, 'sce_sys'))

    # Package .prx libraries
    shutil.copy(
        os.path.join(config.sce_tools_info.sce_orbis_sdk_dir, 'target', 'sce_module', 'libc.prx'),
        os.path.join(config.staging_path, 'sce_module', 'libc.prx'))

    shutil.copy(
        os.path.join(config.sce_tools_info.sce_orbis_sdk_dir, 'target', 'sce_module', 'libSceFios2.prx'),
        os.path.join(config.staging_path, 'sce_module', 'libSceFios2.prx'))

    print('Done staging package...')


def create_package(config):

    print('Creating package description...')

    g4p_project_path = os.path.join(config.staging_path, 'Core.gp4')

    if config.volume_type == 'remaster':
        volume_type = 'pkg_ps4_remaster'
        #volume_type = 'pkg_ps4_app'
        storage_type = 'digital50'
    elif config.volume_type == 'patch':
        volume_type = 'pkg_ps4_patch'
        storage_type = 'digital25'
    else:
        raise Exception('Unhandled volume_type: ' + config.volume_type)

    # Create new project
    shell_args = [config.sce_tools_info.orbis_pub_cmd_path]
    shell_args.extend(['gp4_proj_create'])
    shell_args.extend(['--volume_type', volume_type])
    shell_args.extend(['--app_path', config.master_package_path])
    shell_args.extend(['--latest_pkg_path', config.latest_package_path])
    shell_args.extend(['--storage_type', storage_type])
    shell_args.extend(['--app_type', 'freemium'])
    shell_args.extend(['--app_category', 'gxe'])
    shell_args.extend(['--content_id', config.sce_content_id])
    shell_args.extend(['--passcode', config.sce_passcode])
    shell_args.extend([g4p_project_path])

    returncode = ps4_tools.run_shell_command_and_wait(shell_args)

    if returncode != 0:
        raise Exception('Failed to create project: ' + str(returncode))

    # Add content
    returncode = ps4_tools.run_shell_command_and_wait([config.sce_tools_info.orbis_pub_cmd_path, 'gp4_file_add', '--force', os.path.join(config.staging_path, 'eboot.bin'), 'eboot.bin', g4p_project_path])
    # returncode = ps4_tools.run_shell_command_and_wait([config.sce_tools_info.orbis_pub_cmd_path, 'gp4_file_add', '--force', os.path.join(config.staging_path, 'assets'), 'assets', g4p_project_path])
    returncode = ps4_tools.run_shell_command_and_wait([config.sce_tools_info.orbis_pub_cmd_path, 'gp4_file_add', '--force', os.path.join(config.staging_path, 'sce_sys'), 'sce_sys', g4p_project_path])
    returncode = ps4_tools.run_shell_command_and_wait([config.sce_tools_info.orbis_pub_cmd_path, 'gp4_file_add', '--force', os.path.join(config.staging_path, 'sce_module'), 'sce_module', g4p_project_path])

    # Create .pkg
    returncode = ps4_tools.run_shell_command_and_wait([config.sce_tools_info.orbis_pub_cmd_path, 'img_create', '--oformat', 'pkg+subitem', g4p_project_path, package_dir])

    if returncode != 0:
        raise Exception('Failed to create package: ' + str(returncode))

    print('Done creating package description')


def verify_package(config):

    print('Verifying package...')

    package_name = config.sce_content_id + '-A' + config.sfx_file.get_app_version().replace('.', '') + '-V' + config.sfx_file.get_master_version().replace('.', '') + '.pkg'
    package_path = os.path.join(package_dir, package_name)

    shell_args = [config.sce_tools_info.orbis_pub_cmd_path]
    shell_args.extend(['img_verify'])
    shell_args.extend(['--iformat', 'file'])
    shell_args.extend(['--passcode', config.sce_passcode])
    shell_args.extend(['--format_check', 'on'])
    shell_args.extend(['--integrity_check', 'on'])
    shell_args.extend([package_path])

    returncode = ps4_tools.run_shell_command_and_wait(shell_args)

    if returncode != 0:
        raise Exception('Package verification failed: ' + str(returncode))

    print('Done verifying package...')

def docs_package(config):
    print('Adding docs to submission materials')

    zip_path = glob.glob(os.path.join(package_dir, '*-submission_materials.zip'))
    if len(zip_path) != 1:
        raise Exception('Expected exactly one *-submission_materials.zip file')

    zip_path = zip_path[0]

    patchNotesFullPath = os.path.join(ps4_patchnotes_dir, config.region + '.docx')
    print('Patch Notes path: ' + patchNotesFullPath)

    # Ex: 01.40 -> A0140.docx
    docName = 'A' + config.sfx_file.get_app_version().replace('.', '') + '.docx'

    with zipfile.ZipFile(zip_path, mode='a') as zf:
        zf.write(patchNotesFullPath, arcname=docName, compress_type=zipfile.ZIP_DEFLATED)
        print('Success: appended patch notes to zip file: ' + zip_path)

def generate_package(config):

    print('Generating package...')

    # The staging directory
    if not config.staging_path:
        config.staging_path = os.path.join(package_dir, '_Staging')

    if not os.path.exists(config.staging_path):
        os.makedirs(config.staging_path)

    if not config.master_package_path:

        master_files = config.s3_client().list_objects_v2(Bucket='tvapps-ps4-releasepackages', Prefix=config.region + '/master_01.00')['Contents']

        config.master_package_path, config.master_package_path_size = find_highest_version_package_in_list(master_files)

        if config.master_package_path is None:
            raise Exception('Master package not found')

        print('Using package as master_package_path: ' + config.master_package_path)

        config.master_package_path_local = os.path.realpath(os.path.join(config.package_prefix, config.master_package_path))

        if not os.path.exists(os.path.dirname(config.master_package_path_local)):
            os.makedirs(os.path.dirname(config.master_package_path_local))

        local_size = -1
        try:
            local_size = os.path.getsize(config.master_package_path_local)
        except OSError:
            pass

        if (local_size != config.master_package_path_size):
            print "Downloading %s from S3 ..." %(config.master_package_path)
            config.s3_client().download_file('tvapps-ps4-releasepackages', config.master_package_path, config.master_package_path_local)
        else:
            print "Skipping download, %s already exists locally" %(config.master_package_path)

        config.master_package_path = config.master_package_path_local

    if not config.latest_package_path:

        # Find all directories in this region root and find the one that represents the latest build
        config.latest_package_path = os.path.join(package_dir, 'universal_viewer_ps4_releases', config.region)
        candidate_packages = config.s3_client().list_objects_v2(Bucket='tvapps-ps4-releasepackages', Prefix=config.region)['Contents']
        pattern = re.compile(r'^%s/[^/]+/$' %(config.region))
        candidate_packages = list(filter(lambda x: re.match(pattern, x['Key']), candidate_packages))

        newest = None
        newest_version = -1
        for f in candidate_packages:
            tokens = f['Key'][:-1].split('_')
            if len(tokens) != 2:
                next

            version = float(tokens[1])
            if newest_version < version < float(config.sce_app_version):
                newest_version = version
                newest = f['Key']

        if newest_version == -1:
            raise Exception('Could not find any old versions')

        # Make sure the version is only one ahead. If we are building 01.42, then the old master should be on 01.41
        if not newest_version + .01 == float(config.sce_app_version):
            raise Exception('Patch version must be incremented by .01. Latest version is: ' + str(newest_version) + ' but package version is: ' + "{0:.2f}".format(float(config.sce_app_version)) + ', expecting ' + str(newest_version + 0.01))

        latest_package = config.s3_client().list_objects_v2(Bucket='tvapps-ps4-releasepackages', Prefix=newest)['Contents']

        config.latest_package_path, config.latest_package_path_size = find_highest_version_package_in_list(latest_package)

        if config.latest_package_path is None:
            raise Exception('Latest package not found')

        print('Using package as latest_package_path: ' + config.latest_package_path)

        config.latest_package_path_local = os.path.realpath(os.path.join(config.package_prefix, config.latest_package_path))

        if not os.path.exists(os.path.dirname(config.latest_package_path_local)):
            os.makedirs(os.path.dirname(config.latest_package_path_local))

        local_size = -1
        try:
            local_size = os.path.getsize(config.latest_package_path_local)
        except OSError:
            pass

        if (local_size != config.latest_package_path_size):
            print "Downloading %s from S3 ..." %(config.latest_package_path)
            config.s3_client().download_file('tvapps-ps4-releasepackages', config.latest_package_path, config.latest_package_path_local)
        else:
            print "Skipping download, %s already exists locally" %(config.latest_package_path)

        config.latest_package_path = config.latest_package_path_local

    # Find tools
    if not config.sce_tools_info:
        config.sce_tools_info = ps4_tools.find_sony_tools()

    # Clean previous packages
    clean(config)

    # Verify changeinfo XML files
    verify_change_info(config)

    # Package creation
    stage_package(config)
    create_package(config)

    # Get version info
    if not config.sfx_file:
        param_sfo_path = os.path.join(config.staging_path, 'sce_sys', 'param.sfo')
        config.sfx_file = ps4_tools.SfxFile(config.sce_tools_info)
        config.sfx_file.load_sfo(param_sfo_path)

    # Verify package
    verify_package(config)

    # Move the generated files into the submission directory
    submission_dir = os.path.join(config.output_path, config.region)
    if not os.path.exists(submission_dir):
        os.makedirs(submission_dir)

    # submission material must be ammended with the .docx file
    docs_package(config)

    pkg_file = glob.glob(os.path.join(package_dir, '*.pkg'))
    if len(pkg_file) != 1:
        raise Exception('Expected exactly one package file')
    map_file = glob.glob(os.path.join(package_dir, config.build_config, '*.map'))
    if len(map_file) != 1:
        raise Exception('Expected exactly one map file')
    elf_file = glob.glob(os.path.join(package_dir, config.build_config, '*.elf'))
    if len(elf_file) != 1:
        raise Exception('Expected exactly one elf file')
    zip_path = glob.glob(os.path.join(package_dir, '*-submission_materials.zip'))
    if len(zip_path) != 1:
        raise Exception('Expected exactly one zip file')
    xml_path = glob.glob(os.path.join(package_dir, '*-spec.xml'))
    if len(xml_path) != 1:
        raise Exception('Expected exactly one xml file')

    pkg_file = pkg_file[0]
    map_file = map_file[0]
    elf_file = elf_file[0]
    zip_path = zip_path[0]
    xml_path = xml_path[0]

    shutil.move(pkg_file, os.path.join(submission_dir, os.path.basename(pkg_file)))
    shutil.copy(map_file, os.path.join(submission_dir, os.path.basename(map_file)))
    shutil.copy(elf_file, os.path.join(submission_dir, os.path.basename(elf_file)))
    shutil.move(zip_path, os.path.join(submission_dir, os.path.basename(zip_path)))
    shutil.move(xml_path, os.path.join(submission_dir, os.path.basename(xml_path)))

    # Notify of completion
    print('')
    print('Package generation complete')


# Process the command line arguments if run as the primary script
if __name__ == "__main__":

    # Parse the command line
    config = Configuration()
    parse_command_line(None, config)

    # Generate the package
    generate_package(config)
