#!/usr/bin/python
#
# This a collection of utilities for PS4 scripts
#
# https://ps4.siedev.net/resources/documents/Misc/current/Publishing_Tools_CL-Users_Guide/0003.html

# Import standard modules
import argparse
import base64
import imp
import os
import shutil
import subprocess
import sys
import time
import xml.etree.ElementTree

# Load other scripts for use
tools_dir = os.path.dirname(__file__)#os.path.realpath(os.path.join(os.path.dirname(__file__), '..', '..', 'tools'))
package_dir = os.path.realpath(os.path.join(tools_dir, '..'))
keys = imp.load_source('keys', os.path.join(tools_dir, 'keys.py'))


class SonyToolsConfiguration:

    def __init__(self):
        self.sce_orbis_sdk_dir = None
        self.sce_root_dir = None
        self.orbis_publishing_tools_path = None
        self.orbis_pub_cmd_path = None


class SfxFile:

    def __init__(self, config):
        self.config = config
        self.doc = None

    def load_sfx(self, sfx_path):
        self.doc = xml.etree.ElementTree.parse(sfx_path)

    def load_sfo(self, sfo_path):

        temp_sfx_path = 'temp.sfx'

        shell_args = [self.config.orbis_pub_cmd_path]
        shell_args.extend(['sfo_export'])
        shell_args.extend([sfo_path])
        shell_args.extend([temp_sfx_path])

        print(' '.join(shell_args))

        proc = subprocess.Popen(shell_args)
        proc.wait()

        if proc.returncode != 0:
            raise Exception('Error loading sfo: ' + str(proc.returncode))

        self.load_sfx(temp_sfx_path)

        # Delete the temporary file
        if os.path.isfile(temp_sfx_path):
            os.remove(temp_sfx_path)

    def save_sfx(self, sfx_path):
        self.doc.write(sfx_path)

    def save_sfo(self, sfo_path):

        temp_sfx_path = 'temp.sfx'
        self.save_sfx(temp_sfx_path)

        shell_args = [self.config.orbis_pub_cmd_path]
        shell_args.extend(['sfo_create'])
        shell_args.extend([temp_sfx_path])
        shell_args.extend([sfo_path])

        print(' '.join(shell_args))

        proc = subprocess.Popen(shell_args)
        proc.wait()

        if proc.returncode != 0:
            raise Exception('Error saving sfo: ' + str(proc.returncode))

        # Delete the temporary file
        if os.path.isfile(temp_sfx_path):
            os.remove(temp_sfx_path)

    def set_value(self, key, value):
        root = self.doc.getroot()
        for x in root:
            if x.tag == 'param' and 'key' in x.attrib:
                if x.attrib['key'] == key:
                    x.text = value
                    return

        # Create a new entry
        elem = xml.etree.ElementTree.SubElement(root, 'param', attrib={'key': key})
        elem.text = value

    def get_value(self, key):
        root = self.doc.getroot()
        for x in root:
            if x.tag == 'param' and 'key' in x.attrib:
                if x.attrib['key'] == key:
                    return x.text

        raise Exception("Key not found in sfx '" + key + "'")

    def set_master_version(self, value):
        self.set_value('VERSION', value)

    def get_master_version(self):
        return self.get_value('VERSION')

    def set_app_version(self, value):
        self.set_value('APP_VER', value)

    def get_app_version(self):
        return self.get_value('APP_VER')

    def set_title_id(self, value):
        self.set_value('TITLE_ID', value)

    def get_title_id(self):
        return self.get_value('TITLE_ID')

    def set_content_id(self, value):
        self.set_value('CONTENT_ID', value)

    def get_content_id(self):
        return self.get_value('CONTENT_ID')

    def set_category(self, value):
        self.set_value('CATEGORY', value)

    def get_category(self):
        return self.get_value('CATEGORY')

    def set_remaster_type(self, value):
        self.set_value('REMASTER_TYPE', value)

    def set_parental_level(self, value):
        self.set_value('PARENTAL_LEVEL', value)

    def set_remote_play_key_assign(self, value):
        self.set_value('REMOTE_PLAY_KEY_ASSIGN', value)

    def get_remote_play_key_assign(self):
        return self.get_value('REMOTE_PLAY_KEY_ASSIGN')

    def set_attributes(self, value):
        self.set_value('ATTRIBUTE', value)

    def get_attributes(self):
        return self.get_value('ATTRIBUTE')

    def set_user_defined_param(self, index, value):
        self.set_value('USER_DEFINED_PARAM_' + str(index), str(value))

    def get_user_defined_param(self, index):
        return self.get_value('USER_DEFINED_PARAM_' + str(index))

def find_sony_tools():

    """
    Returns a SonyToolsConfiguration instance.
    """

    result = SonyToolsConfiguration()

    result.sce_orbis_sdk_dir = os.environ["SCE_ORBIS_SDK_DIR"]
    if not result.sce_orbis_sdk_dir:
        raise Exception("SCE_ORBIS_SDK_DIR not set")

    result.sce_root_dir = os.environ["SCE_ROOT_DIR"]
    if not result.sce_root_dir:
        raise Exception("SCE_ROOT_DIR not set")

    if not os.path.isdir(result.sce_orbis_sdk_dir):
        raise Exception(result.sce_orbis_sdk_dir + " not found")

    if not os.path.isdir(result.sce_root_dir):
        raise Exception(result.sce_root_dir + " not found")

    result.orbis_publishing_tools_path = os.path.join(result.sce_root_dir, 'ORBIS', 'Tools', 'Publishing Tools', 'bin')
    result.orbis_pub_cmd_path = os.path.join(result.orbis_publishing_tools_path, 'orbis-pub-cmd.exe')

    return result


def run_shell_command_and_wait(shell_args):

    """
    Prints the command to run. runs the given shell command, waits for it to complete and returns the process exit code
    """

    print(' '.join(shell_args))

    proc = subprocess.Popen(shell_args)
    proc.wait()

    return proc.returncode


def get_regions():

    """
    Returns the list of supported regions.
    """

    return [
        'siea',
        'siee',
        'sieasia'
    ]


def setup_regions_argument(parser):

    """
    Sets up the --region command line argument in the given argparse parser.
    """

    regions = get_regions()

    metavar = '<' + '|'.join(regions) + '>'
    help_text = 'Specifies the region to generate the package for.  Options are ' + ', '.join(regions)

    parser.add_argument(
        '--region',
        required = True,
        metavar = metavar,
        help = help_text,
        choices = regions
    )


def get_region_title_info(region, use_dummy_nptitledat):

    class Result:
        def __init__(self):
            self.sce_title_id = None
            self.sce_content_id = None
            self.sce_passcode = None

    result = Result()
    result.sce_passcode = keys.getSecret('playstation', 'platforms/PlayStation/production/sce_passcode')

    if region == 'siea':
        result.sce_title_id = 'CUSA03285'
        result.sce_content_id = 'UP8902-CUSA03285_00-TWITCHPS4APPSCEA'

    elif region == 'siee':
        result.sce_title_id = 'CUSA03398'
        result.sce_content_id = 'EP8816-CUSA03398_00-TWITCHPS4APPSCEE'

    elif region == 'sieasia':
        result.sce_title_id = 'CUSA03408'
        result.sce_content_id = 'HP8901-CUSA03408_00-TWITCHPS4APPASIA'
    else:
        raise Exception('Unsupported region: ' + region)

    if use_dummy_nptitledat:
        with open(os.path.join(package_dir, 'config', 'template', 'nptitle.dat'), mode='rb') as f:
            result.sce_nptitle_dat = f.read()
    else:
        result.sce_nptitle_dat = base64.b64decode(keys.getSecret('playstation', 'platforms/PlayStation/production/' + result.sce_title_id + '_nptitle'))

    return result
