#!/usr/bin/python
# 
# This script creates a release of the PS4 app
# 
# To build all platforms run the following 3 commands.
#
# Note the --no-clean and --no-build for platforms after the first one.  This is because the app binaries are actually the same.
#
#   python release.py --output-path=C:/Drew/Junk/PS4Builds --build-config=release --region=siea --volume-type=remaster --twitch-build-number=12345 --commit-hash=ae34asdf23cv34
#   python release.py --output-path=C:/Drew/Junk/PS4Builds --build-config=release --region=siee --no-clean --no-build --volume-type=remaster --twitch-build-number=12345 --commit-hash=ae34asdf23cv34
#   python release.py --output-path=C:/Drew/Junk/PS4Builds --build-config=release --region=sieasia --no-clean --no-build --volume-type=remaster --twitch-build-number=12345 --commit-hash=ae34asdf23cv34
#

# Import standard modules
import sys
import os
import argparse
import shutil
import subprocess
import time
import imp

platform_build_dir = os.path.dirname(os.path.realpath(__file__))
application_dir = os.path.realpath(os.path.join(platform_build_dir, '..', '..'))
release_source = imp.load_source('release', os.path.join(application_dir, 'release.py'))
Release = release_source.Release
ps4_package = imp.load_source('ps4_package', os.path.join(platform_build_dir, 'package.py'))

class PlatformRelease(Release):
    def __init__(self):
        self.platform_build_dir = platform_build_dir
        self.application_dir = application_dir

        self.parse_config(platform='ps4', available_build_configs = ['release','final'])

    def build_app(self):
        release_source.call_script_helper('', 'ruby', application_dir,
            ['build.rb',
             '--build_directory', application_dir + '\\build\\ps4',
             '--config', 'release'])
    
    def package_app(self):
        package_config = ps4_package.Configuration()
        if self.config.output_path:
            package_config.output_path = self.config.output_path
        package_config.region = self.config.region
        package_config.volume_type = self.config.volume_type
        package_config.build_config = self.config.build_config
        # package.py uses build_config to find the build directory
        if self.config.build_config == 'final':
            package_config.build_config = 'release'
        package_config.sce_title_id = self.config.sce_title_id
        package_config.sce_master_version = self.config.sce_master_version
        package_config.sce_app_version = self.config.sce_app_version
        package_config.sce_content_id = self.config.sce_content_id
        package_config.sce_passcode = self.config.sce_passcode

        ps4_package.generate_package(package_config)

    def process(self):
        cleaned = False

        if not self.config.no_clean_app:
            cleaned = True
            self.clean_app()

        # Sleep to allow the filesystem to settle
        if cleaned:
            time.sleep(2)

        if not self.config.no_build_app:
            self.build_init()
            print('Generating app...')
            self.generate()

            print('Building app...')
            self.build_app()
        else:
            # Sequential builds for different regions don't require re-building,
            # but they do require re-configuring the app
            print('Generating app...')
            self.generate()

        if not self.config.no_package:
            print('Packaging app...')
            self.package_app()

# Process the command line arguments if run as the primary script
if __name__ == "__main__":
    build = PlatformRelease()
    build.process()
