#!/usr/bin/python
#
# This script downloads the given release notes from smartling and copies them
# into the appropiate changeinfo_xx.xml, there is a different changeinfo_xx.xml
# for each language.
#
# The correct sony product id is copied from the default changeinfo.xml file,
# so make sure that this file exists for each region.
#
# This script writes all translations to every sie region, some of the
# translations will not be used for the given sie region.
#
# Use example:
# python3 sony_package_versioning_update.py --filename=release_notes_4_0_0.txt --platform=ps4 --increment-versions
#
# If the Smartling translation is not ready yet, use the default release notes as a placeholder:
# python3 sony_package_versioning_update.py --filename=default_release_notes.txt --platform=ps4 --increment-versions
#
# If increment-versions is not specified the script will not increment the versions found in versions.json before writing
# the changeinfo.xml files

import sys
assert sys.version_info[0] >= 3

import argparse
from importlib.machinery import SourceFileLoader
import os
import os.path
import json
import sys
import xml.etree.ElementTree as ET
from lxml import etree

# localization = SourceFileLoader("localization", os.path.join(os.path.realpath(os.path.dirname(__file__)), 'localization.py')).load_module()
working_dir = os.path.realpath(os.path.dirname(__file__))
sce_regions = ['siea', 'sieasia', 'siee']
is_debugging = False

def DebugLog(message):
    if is_debugging:
        print("sony_package_versioning_update Log: " + message)

# See https://ps4.siedev.net/resources/documents/SDK/7.500/Content_Information-Specifications/0010.html for specification of language to changeinfo xml
languages_code_xml_map = {
    'cs-CZ':'changeinfo_23.xml',
    'da-DK':'changeinfo_14.xml',
    'de-DE':'changeinfo_04.xml',
    'el-GR':'changeinfo_25.xml',
    'en-US':'changeinfo_01.xml',
    'es-ES':'changeinfo_03.xml',
    'es-MX':'changeinfo_20.xml',
    'fi-FI':'changeinfo_12.xml',
    'fr-CA':'changeinfo_22.xml',
    'fr-FR':'changeinfo_02.xml',
    'hu-HU':'changeinfo_24.xml',
    'id-ID':'changeinfo_29.xml',
    'it-IT':'changeinfo_05.xml',
    'ja-JP':'changeinfo_00.xml',
    'ko-KR':'changeinfo_09.xml',
    'nl-NL':'changeinfo_06.xml',
    'no-NO':'changeinfo_15.xml',
    'pl-PL':'changeinfo_16.xml',
    'pt-BR':'changeinfo_17.xml',
    'pt-PT':'changeinfo_07.xml',
    'ru-RU':'changeinfo_08.xml',
    'sv-SE':'changeinfo_13.xml',
    'th-TH':'changeinfo_27.xml',
    'tr-TR':'changeinfo_19.xml',
    'zh-CN':'changeinfo_11.xml', # Chinese simplified
    'zh-TW':'changeinfo_10.xml'  # Chinese traditional
}

# We need to use lxml library here because the built in element tree doesn't support CDATA
def write_to_xml(path, sce_title_id, sce_version, release_note):
    # if the xml file does not exist create it
    if not(os.path.exists(path)):
        changeinfo = etree.Element('changeinfo')
        changeinfo.attrib['titleid'] = sce_title_id
        changes = etree.SubElement(changeinfo, 'changes')
        changes.attrib['app_ver'] = sce_version
        changes.text = etree.CDATA(release_note)
        root = etree.ElementTree(changeinfo)
        root.write(path, encoding='utf-8', xml_declaration=True, pretty_print=True)

    else:
        parser = etree.XMLParser(strip_cdata=False, remove_blank_text=True)
        doc = etree.parse(path, parser)
        if doc is None:
            raise Exception('Could not load XML doc ' + path)

        xml_changeinfo = doc.getroot()
        assert xml_changeinfo.tag == 'changeinfo'

        # If the changes tag for the version already exists delete it
        for xml_changes in xml_changeinfo.findall('changes'):
            if not 'app_ver' in xml_changes.attrib:
                raise Exception('Could not find app_ver attribute')

            if xml_changes.attrib['app_ver'] == sce_version:
                # Remove the old tag, we are going to add it again below
                xml_changeinfo.remove(xml_changes)

        # Create the new changes tag with the new release notes
        new_changes_tag = etree.Element('changes')
        new_changes_tag.attrib['app_ver'] = sce_version
        new_changes_tag.text = etree.CDATA(release_note)
        xml_changeinfo.insert(0, new_changes_tag)

        doc.write(path, encoding='utf-8', xml_declaration=True, pretty_print=True)

def parse_sce_versions(platform, increment):
    # Parse the default versions.json file to get the sce region version numbers
    root_dir = os.path.join(working_dir, '..')

    full_path = os.path.join(root_dir, 'versions.json')

    sce_version_numbers = list()

    with open(full_path, 'r+') as json_file:
        data = json.load(json_file)

        for region in sce_regions:
            sce_version = data[platform]['sce-app-version-' + region]
            if increment:
                sce_version_number = float(sce_version) + .01
                sce_version = '0' + '{:.2f}'.format(sce_version_number)
                data[platform]['sce-app-version-' + region] = sce_version

            sce_version_numbers.append(sce_version)

        json_file.seek(0)
        json_file.write(json.dumps(data, sort_keys=True, indent=4))
        json_file.write('\n')

    print(sce_version_numbers)
    return sce_version_numbers

def parse_sce_title_id(platform, sce_region):
    # Parse the default changeinfo.xml file to get the sce title id
    root_dir = os.path.join(working_dir, '..', 'application', 'build', platform, 'config')

    full_path = os.path.join(root_dir, sce_region, 'sce_sys', 'changeinfo', 'changeinfo.xml')
    doc = ET.parse(full_path)
    if doc is None:
        raise Exception('Could not load XML doc ' + full_path)

    xml_changeinfo = doc.getroot()
    assert xml_changeinfo.tag == 'changeinfo'

    return xml_changeinfo.attrib['titleid']

def fetch_strings(platform, sce_region, sce_title_id, sce_version, options):
    for language_code in languages_code_xml_map:
        DebugLog('Fetching \"' + language_code + '\" from Smartling...')

        # try:
        #     release_note = localization.fetch_file(options, language_code, options.smartling_file_uri)
        # except Exception as e:
        #         raise e

        # release_note = str.encode(release_note)
        # release_note = release_note.decode('utf-8-sig')
        release_note = "temp_release_note"

        if release_note:
            directory = os.path.join(working_dir, '..', 'application', 'build', platform, 'config', sce_region, 'sce_sys', 'changeinfo')
            path = os.path.join(directory, languages_code_xml_map[language_code])
            write_to_xml(path, sce_title_id, sce_version, release_note)
            # Also write out the default changeinfo.xml with en-US translation
            if language_code == 'en-US':
                default_path = os.path.join(directory, 'changeinfo.xml')
                write_to_xml(default_path, sce_title_id, sce_version, release_note)

        DebugLog('Done fetching \"' + language_code + '\" from Smartling.')

# Process the command line arguments if run as the primary script
if __name__ == "__main__":

    parser = argparse.ArgumentParser(description='Tool to generate sony changeinfo.xml files for all translated languages')

    parser.add_argument(
        '--filename',
        required=True,
        help='Specifies the filename of the translated strings on smartling. Cannot be used with --placeholder'
    )

    parser.add_argument(
        '--platform',
        required=True,
        help='Specifies the name of the platform to geneerate changeinfo xmls for.'
    )

    parser.add_argument(
        '--increment-versions',
        required=False,
        action='store_true',
        default=False,
        help='If true the script will increment the sony region version numbers present in versions.json before writing the changeinfo xmls with the new version number'
    )

    args = parser.parse_args()

    sce_region_to_version = dict(zip(sce_regions, parse_sce_versions(args.platform, args.increment_versions)))

    default_language = "en-US"

    print('Fetching release notes from Smartling and writing changeinfo xml files...')

    for region, sce_version in sce_region_to_version.items():
        sce_title_id = parse_sce_title_id(args.platform, region)
        fetch_strings(args.platform, region, sce_title_id, sce_version, options)

    print('Done Fetching release notes from Smartling and writing changeinfo xml files.')
