#include "app/storagemanager.h"

#include <atomic>

/**
 * Container Organization
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * DeviceData
 *   Written to /download0 with a filename of <containerName>.<key>
 *
 * Blobs
 *   Written to /download0 with a filename of <name>.blob
 *
 * UserData
 *   We discovered that mounting savegames causes latency issues since mounting is an
 *   expensive operation.  Going forward, we create only one shared savegame for all user
 *   data written.  Each access is into the root of the mounted savegame with a
 *   filename of <containerName>.<key>
 *   The format of the data has not changed.
 */

namespace app {
namespace storage {
// Current constants
namespace v1 {
constexpr const char* kSharedContainerName = "tvapps";
/**
 * We allocated a container size of 10MB for all save data.
 */
constexpr size_t kSharedContainerSize = 1024 * 1024 * 10;
}  // namespace v1
}  // namespace storage
}  // namespace app

namespace {
using namespace app;
using namespace app::sce;
using namespace app::storage;

constexpr const char* kLogTag = "StorageManager";

/**
 * This is a directory that is shared between multiple users.  We need to specify the "Download data size" property
 * in the .sfo file to gain access to it.
 */
constexpr const char* kDeviceDataMountPoint = "/download0";
constexpr auto kMountModeWritable = SCE_SAVE_DATA_MOUNT_MODE_RDWR;
constexpr auto kMountModeCreateAndWritable = SCE_SAVE_DATA_MOUNT_MODE_CREATE2 | SCE_SAVE_DATA_MOUNT_MODE_RDWR;

/**
 * Determines the minimum number of blocks needed to store the given data.
 */
SceSaveDataBlocks ComputeNumBlocksNeeded(size_t numBytes) {
  // The number of blocks requested isn't actually the number is allocated.  The OS uses some of the space so we need to over-allocate

  // Round up to the nearest block size
  return static_cast<SceSaveDataBlocks>(
    SCE_SAVE_DATA_BLOCKS_MIN2 + ((static_cast<uint64_t>(numBytes) + 1) / SCE_SAVE_DATA_BLOCK_SIZE) + 1);
}
}  // namespace

namespace app {
namespace storage {
namespace v1 {
void GetUserDataPathComponents(const SceSaveDataMountResult& mount, const std::string& containerName,
  const std::string& key, std::string& directory, std::string& filename) {
  directory = mount.mountPoint.data;
  filename = containerName + "." + key;
}
}  // namespace v1
}  // namespace storage
}  // namespace app

app::sce::StorageManager::StorageManager() : mNumBeginAccesses(0) {
  sceSaveDataInitialize3(nullptr);
}

app::sce::StorageManager::~StorageManager() {
  sceSaveDataTerminate();
}

int32_t app::sce::StorageManager::Initialize() {
  return 0;
}

int32_t app::sce::StorageManager::Terminate() {
  return 0;
}

/**
 * Returns true if an error, false otherwise.
 */
bool app::sce::StorageManager::CheckErrorCode(int32_t ec) {
  switch (ec) {
    case SCE_OK:
      return false;

    case SCE_SAVE_DATA_ERROR_PARAMETER:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_PARAMETER");
      break;
    case SCE_SAVE_DATA_ERROR_NOT_INITIALIZED:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_NOT_INITIALIZED");
      break;
    case SCE_SAVE_DATA_ERROR_OUT_OF_MEMORY:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_OUT_OF_MEMORY");
      break;
    case SCE_SAVE_DATA_ERROR_BUSY:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_BUSY");
      break;
    case SCE_SAVE_DATA_ERROR_EXISTS:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_EXISTS");
      break;
    case SCE_SAVE_DATA_ERROR_NOT_FOUND:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_NOT_FOUND");
      break;
    case SCE_SAVE_DATA_ERROR_NO_SPACE_FS:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_NO_SPACE_FS");
      break;
    case SCE_SAVE_DATA_ERROR_INTERNAL:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_INTERNAL");
      break;
    case SCE_SAVE_DATA_ERROR_MOUNT_FULL:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_MOUNT_FULL");
      break;
    case SCE_SAVE_DATA_ERROR_NOT_MOUNTED:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_NOT_MOUNTED");
      break;
    case SCE_SAVE_DATA_ERROR_BROKEN:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_BROKEN");
      break;
    case SCE_SAVE_DATA_ERROR_INVALID_LOGIN_USER:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_INVALID_LOGIN_USER");
      break;
    case SCE_SAVE_DATA_ERROR_BACKUP_BUSY:
      APP_LOGE(kLogTag, "SCE_SAVE_DATA_ERROR_BACKUP_BUSY");
      break;
    case SCE_USER_SERVICE_ERROR_NOT_INITIALIZED:
      APP_LOGE(kLogTag, "SCE_USER_SERVICE_ERROR_NOT_INITIALIZED");
      break;
    case SCE_KERNEL_ERROR_EBADF:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EBADF");
      break;
    case SCE_KERNEL_ERROR_EFAULT:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EFAULT");
      break;
    case SCE_KERNEL_ERROR_EIO:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EIO");
      break;
    case SCE_KERNEL_ERROR_EINVAL:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EINVAL");
      break;
    case SCE_KERNEL_ERROR_EAGAIN:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EAGAIN");
      break;
    case SCE_KERNEL_ERROR_EISDIR:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EISDIR");
      break;
    case SCE_KERNEL_ERROR_EOPNOTSUPP:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EOPNOTSUPP");
      break;
    case SCE_KERNEL_ERROR_EACCES:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EACCES");
      break;
    case SCE_KERNEL_ERROR_ENAMETOOLONG:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_ENAMETOOLONG");
      break;
    case SCE_KERNEL_ERROR_ENOENT:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_ENOENT");
      break;
    case SCE_KERNEL_ERROR_ENOTDIR:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_ENOTDIR");
      break;
    case SCE_KERNEL_ERROR_EOVERFLOW:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EOVERFLOW");
      break;
    case SCE_KERNEL_ERROR_EROFS:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EROFS");
      break;
    case SCE_KERNEL_ERROR_EMFILE:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EMFILE");
      break;
    case SCE_KERNEL_ERROR_ENFILE:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_ENFILE");
      break;
    case SCE_KERNEL_ERROR_ENOSPC:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_ENOSPC");
      break;
    case SCE_KERNEL_ERROR_EEXIST:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_EEXIST");
      break;
    case SCE_KERNEL_ERROR_ENOBLK:
      APP_LOGE(kLogTag, "SCE_KERNEL_ERROR_ENOBLK");
      break;
  }

  return true;
}

bool app::sce::StorageManager::LoadFile(
  const std::string& directory, const std::string& fileName, std::string& result) {
  result.clear();

  bool error = false;
  int32_t ret = SCE_OK;
  int fd = 0;

  std::string path = directory + "/" + fileName;

  std::string msg = "StorageManager - LoadFile(): Loading file " + path;
  APP_LOGD(kLogTag, msg.c_str());

  // See if the file is there
  SceKernelStat st;
  ret = sceKernelStat(path.c_str(), &st);
  if (CheckErrorCode(ret)) {
    APP_LOGE(kLogTag, "LoadFile-CouldNotFindFile");
    error = true;
  }

  if (error == false) {
    // Allocate memory for the result
    result.resize(static_cast<size_t>(st.st_size));

    // Open the file
    fd = sceKernelOpen(path.c_str(), SCE_KERNEL_O_RDONLY, SCE_KERNEL_S_INONE);
    if (fd < SCE_OK) {
      int32_t ec = static_cast<int32_t>(fd);
      if (CheckErrorCode(ec)) {
        APP_LOGE(kLogTag, "LoadFile-CouldNotOpenFile");
        error = true;
      }
      result.resize(0);
      fd = 0;
    }
  }

  if (error == false) {
    // Read the file
    ssize_t numRead = sceKernelRead(fd, reinterpret_cast<void*>(const_cast<char*>(result.data())), result.size());
    if (numRead < SCE_OK) {
      int32_t ec = static_cast<int32_t>(numRead);
      if (CheckErrorCode(ec)) {
        APP_LOGE(kLogTag, "LoadFile-CouldNotReadFile");
        error = true;
      }
      result.resize(0);
    }
  }

  // Close the file
  if (fd != 0) {
    int32_t ec = sceKernelClose(fd);
    if (CheckErrorCode(ec)) {
      APP_LOGE(kLogTag, "LoadFile-CouldNotCloseFile");
      error = true;
    }
  }

  APP_LOGD(kLogTag, "StorageManager - LoadFile(): done");

  return error;
}

/**
 * Deletes the given save game.  Normally used for legacy cleanup.
 */
bool app::sce::StorageManager::DeleteContainer(SceUserServiceUserId userId, const std::string& containerName) {
  int32_t ec = SCE_OK;
  bool error = false;

  SceSaveDataDirName dirName;
  APP_ASSERT(containerName.size() < sizeof(dirName.data) - 1, kLogTag, "Container name too long");
  snprintf(dirName.data, sizeof(dirName.data), "%s", containerName.c_str());

  SceSaveDataDelete config;
  memset(&config, 0, sizeof(config));
  config.userId = userId;
  config.dirName = &dirName;

  // Delete
  APP_LOGD(kLogTag, "sceSaveDataDelete");
  ec = sceSaveDataDelete(&config);

  if (CheckErrorCode(ec)) {
    APP_LOGE(kLogTag, "DeleteContainer-FailedToDelete");
    error = true;
  }

  return error;
}

/**
 * Safely writes a file
 * When writing a file, a processing for example temporarily creating the file with a different name and then
 * changing it to the formal name after completing writing is required.
 * Although the fileName passed with an argument is the formal file name in this function,
 * "TEMP0_" + fileName and "TEMP1_" + fileName are set as the temporary file names as the transaction processing.
 */
bool app::sce::StorageManager::WriteFileWithTransaction(
  const std::string& directory, const std::string& fileName, const std::string& data) {
  bool error = false;

  // Prepare paths
  std::string filePath = directory + "/" + fileName;
  std::string previousTempPath = filePath + ".last";
  std::string newTempPath = filePath + ".temp";

  // Delete temporary files unconditionally
  sceKernelUnlink(previousTempPath.c_str());
  sceKernelUnlink(newTempPath.c_str());

  // Write the new data to a temporary file
  int fd = sceKernelOpen(
    newTempPath.c_str(), (SCE_KERNEL_O_RDWR | SCE_KERNEL_O_TRUNC | SCE_KERNEL_O_CREAT), SCE_KERNEL_S_IRWU);
  if (fd < SCE_OK) {
    int32_t ec = static_cast<int32_t>(fd);
    if (CheckErrorCode(ec)) {
      APP_LOGE(kLogTag, "StorageManager::WriteFileWithTransaction-CouldNotCreateNewTempFile");
      error = true;
    }
    fd = 0;
  }

  if (error == false) {
    ssize_t numWritten = static_cast<int32_t>(sceKernelWrite(fd, data.c_str(), data.size()));
    if (numWritten < SCE_OK) {
      int32_t ec = static_cast<int32_t>(numWritten);
      if (CheckErrorCode(ec)) {
        APP_LOGE(kLogTag, "WriteFileWithTransaction-CouldNotWriteNewTempFile");
        error = true;
      }
    }
  }

  if (fd != 0) {
    int32_t ec = sceKernelClose(fd);
    if (CheckErrorCode(ec)) {
      APP_LOGE(kLogTag, "WriteFileWithTransaction-CouldNotCloseFile");
      error = true;
    }
    fd = 0;
  }

  // Rename existing file name
  if (error == false) {
    int ret = sceKernelRename(filePath.c_str(), previousTempPath.c_str());
    if (ret != SCE_OK && ret != SCE_KERNEL_ERROR_ENOENT) {
      int32_t ec = static_cast<int32_t>(fd);
      if (CheckErrorCode(ec)) {
        APP_LOGE(kLogTag, "WriteFileWithTransaction-CouldNotRenamePreviousFile");
        error = true;
      }
    }
  }

  // Change the file name written with temporary name to the formal one
  if (error == false) {
    int ret = sceKernelRename(newTempPath.c_str(), filePath.c_str());
    if (ret != SCE_OK) {
      int32_t ec = static_cast<int32_t>(fd);
      if (CheckErrorCode(ec)) {
        APP_LOGE(kLogTag, "WriteFileWithTransaction-CouldNotRenameNewFile");
        error = true;
      }
    }
  }

  return error;
}

bool app::sce::StorageManager::Mount(const MountParams& params, SceSaveDataMountResult& result) {
  bool error = false;
  {
    // Force creation by deleteing the shared container
    // https://ps4.siedev.net/resources/documents/SDK/7.000/SaveData-Overview/0005.html#__document_toc_00000025
#if 0
    static bool once = true;
    if (once) {
      DeleteContainer(params.userId, v1::kSharedContainerName);
      once = false;
    }
#endif

    // Mount
    int32_t mountEc = SceMount(params, result);

    if (CheckErrorCode(mountEc)) {
      APP_LOGE(kLogTag, "Mount-FailedToMount");
      error = true;

      // Container is corrupted, delete it
      if (mountEc == SCE_SAVE_DATA_ERROR_BROKEN) {
        std::string msg = "Save game container is corrupted, deleting: " + params.containerName;
        APP_LOGE(kLogTag, msg.c_str());
        bool deleteError = DeleteContainer(params.userId, params.containerName);
        if (deleteError == true) {
          APP_LOGE(kLogTag, "Mount-CorruptDataDeleteFailed");
        }

        // Display message to user informing them that the corrupted data has been deleted & recreated
        HandleCorruptData(params.userId);
      }
      // Container not found
      else if (mountEc == SCE_SAVE_DATA_ERROR_NOT_FOUND) {
        std::string msg = "Save game container not found: " + params.containerName;
        APP_LOGE(kLogTag, msg.c_str());
      }

      // Create the container
      if (params.allowRecreationAttempt) {
        MountParams createParams;
        createParams.userId = params.userId;
        createParams.containerName = v1::kSharedContainerName;
        createParams.mountMode = kMountModeCreateAndWritable;
        createParams.numBlocks = ComputeNumBlocksNeeded(v1::kSharedContainerSize);
        createParams.transactionResourceId = GetTransactionResourceId();
        createParams.allowRecreationAttempt = true;

        auto createEc = SceMount(createParams, result);
        if (CheckErrorCode(createEc)) {
          APP_LOGE(kLogTag, "Mount-FailedToCreateSaveData");
          error = true;
          // TODO: <rjh> keeping this info could be useful
          // if (storageError != nullptr) {
          //   storageError->SetNumBlocks(params.numBlocks);
          //   storageError->SetContainerName(params.containerName);
          // }
        } else {
          error = false;
        }
      }
    }
  }
  return error;
}

bool app::sce::StorageManager::Unmount(const SceSaveDataMountResult& mount) {
  bool error = false;

  // Unmount
  int32_t ec = SceUnmount(mount.mountPoint);
  if (CheckErrorCode(ec)) {
    APP_LOGE(kLogTag, "Unmount-FailedToUnmount");
    error = true;
  }

  return error;
}

bool app::sce::StorageManager::SafeMount(
  const MountParams& params, const std::function<bool(const SceSaveDataMountResult&)>& callback) {
  bool error = false;
  bool mounted = false;
  SceSaveDataMountResult result;

  error = Mount(params, result);

  // Do the work
  if (error == false) {
    mounted = true;
    error = callback(result);
  }

  // Unmount
  if (mounted) {
    bool unmountError = Unmount(result);
    if (error == false) {
      error = unmountError;
    }
  }

  return error;
}

bool app::sce::StorageManager::SetDeviceData(
  const std::string& containerName, const std::string& key, const json11::Json::object& data) {
  std::string fileName = containerName + "." + key;
  auto serializedData = json11::Json(data).dump();

  APP_LOGD(kLogTag, "SetDeviceData >> Writing to file");
  return WriteFileWithTransaction(kDeviceDataMountPoint, fileName, serializedData);
}

bool app::sce::StorageManager::GetDeviceData(
  const std::string& containerName, const std::string& key, json11::Json result) {
  std::string fileName = containerName + "." + key;
  std::string seralizedResult;
  std::string errorMessage;

  bool error = LoadFile(kDeviceDataMountPoint, fileName, seralizedResult);

  if (error == false) {
    result = json11::Json::parse(seralizedResult, errorMessage, json11::JsonParse::STANDARD);
    if (!errorMessage.empty()) {
      error = true;
    }
  }

  return error;
}

bool app::sce::StorageManager::HandleCorruptData(SceUserServiceUserId userId) {
  SceSaveDataDialogParam params;
  sceSaveDataDialogParamInitialize(&params);

  SceSaveDataDialogSystemMessageParam msgParam;
  memset(&msgParam, 0, sizeof(msgParam));
  msgParam.sysMsgType = SCE_SAVE_DATA_DIALOG_SYSMSG_TYPE_CORRUPTED_AND_CREATE;

  SceSaveDataDialogItems items;
  memset(&items, 0, sizeof(items));
  items.userId = userId;

  params.mode = SCE_SAVE_DATA_DIALOG_MODE_SYSTEM_MSG;
  params.dispType = SCE_SAVE_DATA_DIALOG_TYPE_SAVE;
  params.sysMsgParam = &msgParam;
  params.items = &items;

  if (sceCommonDialogIsUsed()) {
    sceMsgDialogTerminate();
  }

  sceSaveDataDialogInitialize();
  int32_t ec = sceSaveDataDialogOpen(&params);
  if (ec != SCE_OK) {
    APP_LOGE(kLogTag, "HandleCorruptData()-FailedToOpenDialog");
    return false;
  }

  sceSaveDataDialogTerminate();

  return true;
}

// AsyncResultPtr<void> app::sce::StorageManager::HandleNoSpace(
//   const std::shared_ptr<PlatformUser>& user, const std::string& containerName, uint64_t numBlocks) {
//   ps4::Ps4App* ps4App = static_cast<ps4::Ps4App*>(GetApplication());
//
//   auto finalAr = GetAsyncResultTracker()->CreateAsyncResult<void>(AR_LOC);
//
//   MountParams mountParams;
//   mountParams.userId = user->GetUserId();
//   mountParams.containerName = containerName;
//   mountParams.mountMode = kMountModeCreateAndWritable;
//   mountParams.numBlocks = numBlocks;
//   mountParams.transactionResourceId = GetTransactionResourceId();
//
//   // Attempt to mount the same size of data to get the required number of blocks from the OS
//   SceSaveDataDirName dirName;
//   APP_ASSERT(containerName.size() < sizeof(dirName.data) - 1, kLogTag, "Container name too long");
//   snprintf(dirName.data, sizeof(dirName.data), "%s", containerName.c_str());
//
//   SceSaveDataMountResult result;
//
//   int32_t ec = SceMount(mountParams, result);
//
//   if (ec != SCE_SAVE_DATA_ERROR_NO_SPACE_FS) {
//     if (ec == SCE_OK) {
//       SceUnmount(result.mountPoint);
//     }
//
//     finalAr->Reject(
//       std::make_shared<ApiError>("StorageManager::HandleNoSpace()-MountErrorWasNotNoSpace", ec));
//     return finalAr;
//   }
//
//   SceSaveDataDialogParam params;
//   sceSaveDataDialogParamInitialize(&params);
//
//   SceSaveDataDialogSystemMessageParam msgParam;
//   memset(&msgParam, 0, sizeof(msgParam));
//   msgParam.sysMsgType = SCE_SAVE_DATA_DIALOG_SYSMSG_TYPE_NOSPACE;
//   msgParam.value = result.requiredBlocks;
//
//   SceSaveDataDialogItems items;
//   memset(&items, 0, sizeof(items));
//   items.userId = user->GetUserId();
//
//   params.mode = SCE_SAVE_DATA_DIALOG_MODE_SYSTEM_MSG;
//   params.dispType = SCE_SAVE_DATA_DIALOG_TYPE_SAVE;
//   params.sysMsgParam = &msgParam;
//   params.items = &items;
//
//   if (sceCommonDialogIsUsed()) {
//     sceMsgDialogTerminate();
//   }
//
//   sceSaveDataDialogInitialize();
//
//   ec = sceSaveDataDialogOpen(&params);
//
//   auto dialogAr = GetAsyncResultTracker()->CreateAsyncResult<int>(AR_LOC);
//   ps4App->GetApiPoller()->WaitForSaveDialog(dialogAr);
//
//   dialogAr->Then<void>(AR_LOC, [finalAr, user, mountParams, result, this](int /*dialogResult*/) {
//     sceSaveDataDialogTerminate();
//
//     auto mountResult = result;
//     memset(&mountResult, 0, sizeof(result));
//
//     auto ec = SceMount(mountParams, mountResult);
//
//     if (ec == SCE_SAVE_DATA_ERROR_NO_SPACE_FS) {
//       // The user hasn't made enough space - loop again on the dialog
//       finalAr->Reject(
//         std::make_shared<ApiError>("StorageManager::HandleNoSpace()-UserDidntMakeSpace", ec));
//     } else if (ec == SCE_OK) {
//       SceUnmount(result.mountPoint);
//       finalAr->Resolve();
//     } else {
//       finalAr->Resolve();
//     }
//   });
//
//   return finalAr;
// }

bool app::sce::StorageManager::SetUserData(const std::shared_ptr<PlaystationPlatformUser>& playstationUser,
  const std::string& containerName, const std::string& key, const json11::Json& data) {
  // Encode the data
  std::string serializedData = json11::Json(data).dump();

  auto func = [this, key, &containerName, &serializedData](const SceSaveDataMountResult& result) {
    std::string directory;
    std::string filename;
    v1::GetUserDataPathComponents(result, containerName, key, directory, filename);
    bool error = false;

    // Prepare mountpoint for writing (this is a no op on Ps4)
    auto ec = ScePrepare(result.mountPoint);
    if (ec < SCE_OK) {
      if (CheckErrorCode(ec)) {
        APP_LOGE(kLogTag, "WriteFileWithTransaction-CouldNotPrepareNewTempFileForWriting");
        error = true;
      }
    }

    // Write file
    if (error == false) {
      error = WriteFileWithTransaction(directory, filename, serializedData);
    }

    // Commit write (this is a no op on Ps4)
    if (error == false) {
      int32_t ec = SceCommit();
      if (ec < SCE_OK) {
        if (CheckErrorCode(ec)) {
          APP_LOGE(kLogTag, "WriteFileWithTransaction-CouldNotCommitNewTempFile");
          error = true;
        }
      }
    }

    return error;
  };

  bool error = false;
  if (mSharedContainerMountedUser == playstationUser && mSharedContainerMountData.mountPoint.data[0] != '\0') {
    error = func(mSharedContainerMountData);
  } else {
    MountParams params;
    params.userId = playstationUser->GetUserId();
    params.containerName = v1::kSharedContainerName;
    params.numBlocks = ComputeNumBlocksNeeded(v1::kSharedContainerSize);
    params.mountMode = kMountModeWritable;
    params.allowRecreationAttempt = true;
    params.transactionResourceId = GetTransactionResourceId();

    error = SafeMount(params, func);
  }

  if (error == true) {
    // TODO: <rjh> log this information
    // storageError->SetPlatformUser(playstationUser);
    // storageError->SetContainerName(containerName);
    // storageError->SetKeyName(key);
  }

  return error;
}

bool app::sce::StorageManager::GetUserData(const std::shared_ptr<PlaystationPlatformUser>& playstationUser,
  const std::string& containerName, const std::string& key, json11::Json result) {
  auto func = [this, &containerName, &key, &result](const SceSaveDataMountResult& mount) {
    std::string directory;
    std::string filename;
    std::string seralizedResult;
    std::string errorMessage;

    v1::GetUserDataPathComponents(mount, containerName, key, directory, filename);

    bool error = LoadFile(directory, filename, seralizedResult);
    if (error == false) {
      result = json11::Json::parse(seralizedResult, errorMessage, json11::JsonParse::STANDARD);
      if (!errorMessage.empty()) {
        error = true;
      }
    }

    return error;
  };

  bool error = false;
  if (mSharedContainerMountedUser == playstationUser && mSharedContainerMountData.mountPoint.data[0] != '\0') {
    error = func(mSharedContainerMountData);
  } else {
    MountParams params;
    params.userId = playstationUser->GetUserId();
    params.containerName = v1::kSharedContainerName;
    params.mountMode = kMountModeWritable;
    params.numBlocks = ComputeNumBlocksNeeded(v1::kSharedContainerSize);
    params.transactionResourceId = GetTransactionResourceId();
    params.allowRecreationAttempt = true;

    error = SafeMount(params, func);
  }

  if (error == true) {
    // storageError->SetPlaystationPlatformUser(playstationUser);
    // storageError->SetContainerName(containerName);
    // storageError->SetKeyName(key);
  }

  return error;
}

// ErrorPtr app::sce::PlaystationStorageManager::HandleNoSpaceError(
//   SceUserServiceUserId userId, const std::string& containerName, uint64_t numBlocks) {
//   // We need to re-mount in order to see how much space we need.
//   int32_t ec = SCE_OK;
//
//   MountParams mountParams;
//   mountParams.userId = userId;
//   mountParams.containerName = containerName;
//   mountParams.mountMode = kMountModeCreateAndWritable;
//   mountParams.numBlocks = numBlocks;
//   mountParams.transactionResourceId = GetTransactionResourceId();
//
//   SceSaveDataMountResult result;
//
//   ec = SceMount(mountParams, result);
//
//   if (ec != SCE_SAVE_DATA_ERROR_NO_SPACE_FS) {
//     if (ec == SCE_OK) {
//       SceUnmount(result.mountPoint);
//     }
//
//     return std::make_shared<PlaystationApiError>("PlaystationStorageManager::HandleNoSpaceError()-NotNoSpaceError", ec);
//   }
//
//   SceSaveDataDialogParam params;
//   sceSaveDataDialogParamInitialize(&params);
//
//   SceSaveDataDialogSystemMessageParam msgParam;
//   memset(&msgParam, 0, sizeof(SceSaveDataDialogSystemMessageParam));
//   msgParam.sysMsgType = SCE_SAVE_DATA_DIALOG_SYSMSG_TYPE_NOSPACE;
//   msgParam.value = result.requiredBlocks;
//
//   SceSaveDataDialogItems items;
//   memset(&items, 0, sizeof(items));
//   items.userId = userId;
//
//   params.mode = SCE_SAVE_DATA_DIALOG_MODE_SYSTEM_MSG;
//   params.dispType = SCE_SAVE_DATA_DIALOG_TYPE_SAVE;
//   params.sysMsgParam = &msgParam;
//   params.items = &items;
//
//   SetupSaveDataDialog();
//   ec = sceSaveDataDialogOpen(&params);
//
//   if (ec != SCE_OK) {
//     return std::make_shared<PlaystationApiError>(
//       "PlaystationStorageManager::HandleNoSpaceError()-FailedToOpenNoSpaceDialog", ec);
//   }
//
//   while (true) {
//     if (sceSaveDataDialogUpdateStatus() == SCE_COMMON_DIALOG_STATUS_FINISHED) {
//       sceSaveDataDialogTerminate();
//       break;
//     }
//   }
//
//   ec = SceMount(mountParams, result);
//
//   if (ec == SCE_SAVE_DATA_ERROR_NO_SPACE_FS) {
//     return HandleNoSpaceError(userId, containerName, numBlocks);
//   } else if (ec != SCE_OK) {
//     return std::make_shared<PlaystationApiError>(
//       "PlaystationStorageManager::HandleNoSpaceError()-NewErrorAfterUserSpaceCleared", ec);
//   }
//
//   ec = SceUnmount(result.mountPoint);
//
//   if (ec != SCE_OK) {
//     return std::make_shared<PlaystationApiError>(
//       "PlaystationStorageManager::HandleNoSpaceError()-SecondUnmountFailed", ec);
//   }
//
//   return nullptr;
// }

// std::shared_ptr<ttv::TaskRunner> app::sce::PlaystationStorageManager::GetTaskRunnerForContainer(
//   const std::string& container) {
//   ttv::AutoMutex lock(mTaskRunnerMutex.get());
//
//   auto iter = mTaskRunners.find(container);
//   if (iter != mTaskRunners.end()) {
//     return iter->second;
//   } else {
//     auto taskRunner = GetTwitchSdkManager()->CreateTaskRunner();
//     mTaskRunners[container] = taskRunner;
//     return taskRunner;
//   }
// }
