# ensure Python3
import sys
assert sys.version_info[0] >= 3

import argparse
import os
import subprocess
import sys


def run_command(input):
    process = subprocess.Popen(input, stdout=subprocess.PIPE)
    stdout, stderr = process.communicate()
    if not stdout is None:
        stdout = stdout.decode()
    if not stderr is None:
        stderr = stderr.decode()
    if process.returncode != 0:
        sys.exit(process.returncode)
    return stdout.splitlines()


def get_filter_files(file_paths):
    included_filter = ['native-apps/nintendo-switch', 'native-apps/common', 'native-apps/playstation']
    excluded_filter = []

    return [
        x for x in file_paths
        if x.startswith(tuple(included_filter)) and not any(
            f in x for f in excluded_filter)
    ]


extensions = ['*.h', '*.hpp', '*.inl', '*.c', '*.cpp', '*.m', '*.mm', '*.java']

def get_modified_files():
    return get_filter_files(
        # include files that have been modified + new files
        run_command(['git', 'diff', '--diff-filter=d', '--name-only', 'HEAD'] + extensions) +
        run_command(['git', 'ls-files', '--other', '--exclude-standard'] + extensions))


def get_files_from_git_hash(git_hashes):
    
    found_files = []
    for hash in git_hashes:
        paths = run_command([
            'git', 'diff-tree', '--no-commit-id', '--name-only', '-r', hash
        ])
        for path in paths:
            if not path in found_files:
                found_files.append(path)

    return get_filter_files(found_files)


def get_all_files():
    return get_filter_files(
        run_command(
            ['git', 'ls-files'] + extensions))


def wait_for_tasks(tasks, max_size):
    completed = [task for task in tasks if task.poll()]
    for c in completed:
        c.wait()
        tasks.remove(c)

    while len(tasks) > max_size:
        tasks[0].wait()
        tasks.pop(0)


def format(all_files, git_hashes):
    # set path to current file
    os.chdir(os.path.dirname(os.path.realpath(__file__)))

    # we can now run git to get root path and set it as our new working path
    os.chdir(run_command(['git', 'rev-parse', '--show-toplevel'])[0])
    print(os.getcwd())

    # run the local clang-format to ensure consistency
    if (all_files):
        git_files = get_all_files()
    elif git_hashes:
        git_files = get_files_from_git_hash(git_hashes)
    else:
        git_files = get_modified_files()

    tasks = []
    for filtered_file in git_files:
        abs_filtered_file = os.path.abspath(filtered_file)
        print(abs_filtered_file)

        tasks.append(
            subprocess.Popen(['clang-format', '-i', abs_filtered_file],
                             stdout=subprocess.PIPE))
        wait_for_tasks(tasks, 10)

    wait_for_tasks(tasks, 0)


def main():
    p = argparse.ArgumentParser(
        description='runs clang-format on project files')

    p.add_argument(
        '-a',
        '--all',
        action='store_true',
        help=
        'format all files valid files in git, default is to run on only the modified files'
    )

    p.add_argument(
        '--hash',
        action='append',
        dest='git_hashes',
        default=[],
        help=
        'format all files that were listed in the given git commit hash'
    )

    opts = p.parse_args()

    format(opts.all, opts.git_hashes)

if __name__ == '__main__':
    main()
