#!/usr/bin/env node

/**
 * This script converts the messages downloaded using intl:download
 * into the format and directory structure required by LG.
 */

const {
  ensureDirSync,
  outputJsonSync,
  readdirSync,
  readJsonSync,
} = require('fs-extra');
const { join } = require('path');

const srcDir = join(__dirname, '../src');
const resourcesDir = join(srcDir, 'resources');

const messagesDir = join(__dirname, '../messages');

// Create resources directory
ensureDirSync(resourcesDir);

const descriptionNamespace = '[AppDescription]';
let englishTranslation = '';
// Source canonical english translation from the root `appDescription.json` file.
englishTranslation = getDescriptionFrom(join(__dirname, '../appDescription.json'));

/**
 * @param filePath {string}
 */
function getDescriptionFrom(filePath) {
  const messages = readJsonSync(filePath);
  // Use the message namespace to find the proper key in case we add more translations later
  const entry = Object.entries(messages).find(([key]) =>
    key.includes(descriptionNamespace),
  );
  if (!entry) {
    // Fall back to english if not found
    return englishTranslation;
  }
  return entry[1];
}

// Iterate over all locales and output to resources dir
readdirSync(messagesDir).forEach((directoryName) => {
  const dirParts = directoryName.split('-');
  // Uppercase the region if it exists
  if (dirParts[1]) {
    dirParts[1] = dirParts[1].toUpperCase();
  }
  const localeDir = join(resourcesDir, ...dirParts);
  ensureDirSync(localeDir);
  const appDescription = getDescriptionFrom(
    join(messagesDir, directoryName, 'messages.json'),
  );

  outputJsonSync(
    join(localeDir, 'appInfo.json'),
    {
      appDescription,
      title: 'Twitch',
    },
    { spaces: 2 },
  );
});

// Modify the root appInfo.json file app description so that it is always up to date
const rootAppInfoFile = join(srcDir, 'appInfo.json');
const rootAppInfo = readJsonSync(rootAppInfoFile);
rootAppInfo.appDescription = englishTranslation;
outputJsonSync(rootAppInfoFile, rootAppInfo, { spaces: 2 });
