import type { FC } from 'react';
import { useState } from 'react';
import ReactDOM from 'react-dom';
import styled from 'styled-components';
import type { Minixperiments } from 'tachyon-experiments';
import {
  Control,
  Experiment,
  ExperimentGroup,
  ExperimentsRoot,
  Treatment,
} from 'tachyon-experiments';
import { CoreUiSsrRoot } from 'tachyon-more-ui';
import { Button } from 'twitch-core-ui';

const TEST_EXPERIMENT = 'TEST_EXPERIMENT_UUID';
const TEST_MULTI_BUCKET_EXPERIMENT = 'TEST_MULTI_BUCKET_UUID';
const TEST_ANOTHER_EXPERIMENT = 'TEST_ANOTHER_UUID';

const ScContent = styled.div`
  display: flex;
`;

const ScText = styled.p`
  margin-bottom: 2rem;
`;

const ScCenteredText = styled(ScText)`
  text-align: center;
`;

const ScMain = styled.main`
  max-height: 85vh;
  overflow-y: scroll;
  padding: 2rem;
`;

const ScAside = styled.aside`
  background: papayawhip;
  color: maroon;
  min-width: 250px;
  padding: 2rem;
`;

const ScFieldSet = styled.fieldset`
  margin-bottom: 1rem;
`;

const ScExperimentButton = styled(Button)`
  display: block;
  margin-bottom: 0.2em;

  &:not(:last-child) {
    margin-right: 0.5rem;
  }
`;

const ScSection = styled.section`
  border: 1px solid #bbb;
  padding: 2rem;
`;

const ScRandomImage = styled.img`
  height: 300px;
  width: 200px;
`;

const ScNestedExperiment = styled.div`
  background: olivedrab;
  border: 1px solid black;
  font-size: 24px;
  font-weight: bold;
  margin-top: 2rem;
  padding: 1rem;
  text-align: center;

  ${ScText} {
    border-bottom: 1px dotted black;
  }
`;

function useExperimentGroup(): [
  NonNullable<Minixperiments[string]>['groups'],
  (groups: ExperimentGroup) => void,
] {
  const [state, setState] = useState([
    {
      value: ExperimentGroup.Control,
      weight: 100,
    },
    {
      value: ExperimentGroup.Treatment,
      weight: 0,
    },
  ]);

  return [
    state,
    (group: ExperimentGroup) => {
      const otherGroup =
        group === ExperimentGroup.Control
          ? ExperimentGroup.Treatment
          : ExperimentGroup.Control;

      setState([
        { value: group, weight: 100 },
        { value: otherGroup, weight: 0 },
      ]);
    },
  ];
}

const Example: FC = () => {
  const [outerExperimentGroup, updateOuterExperiment] = useExperimentGroup();
  const [innerExperimentGroup, updateInnerExperiment] = useExperimentGroup();
  const [anotherExperimentGroup, updateAnotherExperiment] =
    useExperimentGroup();

  return (
    <CoreUiSsrRoot appRootElementId="root">
      <ScCenteredText as="h1">Experiment Example</ScCenteredText>
      <ScContent>
        <ScAside>
          <ScFieldSet>
            <legend>Outer experiment:</legend>
            <ScExperimentButton
              onClick={() => updateOuterExperiment(ExperimentGroup.Control)}
            >
              Control
            </ScExperimentButton>
            <ScExperimentButton
              onClick={() => updateOuterExperiment(ExperimentGroup.Treatment)}
            >
              Treatment
            </ScExperimentButton>
          </ScFieldSet>
          <ScFieldSet>
            <legend>Inner experiment:</legend>
            <ScExperimentButton
              onClick={() => updateInnerExperiment(ExperimentGroup.Control)}
            >
              Control
            </ScExperimentButton>
            <ScExperimentButton
              onClick={() => updateInnerExperiment(ExperimentGroup.Treatment)}
            >
              Treatment
            </ScExperimentButton>
          </ScFieldSet>
          <ScFieldSet>
            <legend>Another experiment:</legend>
            <ScExperimentButton
              onClick={() => updateAnotherExperiment(ExperimentGroup.Control)}
            >
              Control
            </ScExperimentButton>
            <ScExperimentButton
              onClick={() => updateAnotherExperiment(ExperimentGroup.Treatment)}
            >
              Treatment
            </ScExperimentButton>
          </ScFieldSet>
        </ScAside>
        <ScMain>
          <ScSection>
            <ScText>
              The experiment framework works by taking a bucket ID for
              experiments.
            </ScText>
          </ScSection>
          <ExperimentsRoot
            bucket="a"
            experimentOverrides={undefined}
            experimentSlots={{
              0: TEST_ANOTHER_EXPERIMENT,
              1: TEST_EXPERIMENT,
              2: TEST_MULTI_BUCKET_EXPERIMENT,
            }}
            experiments={{
              [TEST_ANOTHER_EXPERIMENT]: {
                groups: anotherExperimentGroup,
                name: 'mobile_web_test_multi_bucket',
                t: 1,
                v: 1,
              },
              [TEST_EXPERIMENT]: {
                groups: outerExperimentGroup,
                name: 'mobile_web_test',
                t: 1,
                v: 4858,
              },
              [TEST_MULTI_BUCKET_EXPERIMENT]: {
                groups: innerExperimentGroup,
                name: 'mobile_web_test_multi_bucket',
                t: 1,
                v: 4859,
              },
            }}
            onEvent={console.log}
          >
            <ScSection>
              <Experiment experimentUUID={TEST_EXPERIMENT}>
                <Control>
                  <ScText>
                    You are in the <strong>control</strong> group! Here is a
                    random picture for your troubles:
                  </ScText>
                  <ScRandomImage src="https://picsum.photos/200/300/?random" />
                  <ScNestedExperiment>
                    <ScText as="h2">
                      What is this hiding down here? Another experiment?
                    </ScText>
                    <Experiment experimentUUID={TEST_MULTI_BUCKET_EXPERIMENT}>
                      <Control>So much control</Control>
                      <Treatment>
                        At least we are getting something experimental
                      </Treatment>
                    </Experiment>
                  </ScNestedExperiment>
                </Control>
                <Treatment>
                  <ScText>
                    You are in the <strong>treatment</strong> group! Here is a
                    random grayscale picture for your troubles:
                  </ScText>
                  <ScRandomImage src="https://picsum.photos/g/200/300/?random" />
                </Treatment>
              </Experiment>
            </ScSection>
            <ScSection>
              <ScText as="h2">Another experiment!</ScText>
              <Experiment experimentUUID={TEST_ANOTHER_EXPERIMENT}>
                <Control>
                  <ScText>
                    You are in the <strong>control</strong> group!
                  </ScText>
                </Control>
                <Treatment>
                  <ScText>
                    You are in the <strong>treatment</strong> group!
                  </ScText>
                </Treatment>
              </Experiment>
            </ScSection>
          </ExperimentsRoot>
        </ScMain>
      </ScContent>
    </CoreUiSsrRoot>
  );
};

ReactDOM.render(<Example />, document.getElementById('root'));
