import type { FC } from 'react';
import type {
  InteractionDataPayload,
  InteractionTrackingProps,
} from 'tachyon-event-tracker';
import {
  ExtendInteractionMedium,
  InteractionType,
  useInteractionTracking,
  withInteractionTracking,
} from 'tachyon-event-tracker';
import { Button, ButtonType, Layout } from 'twitch-core-ui';
import { ScExampleSection, ScText } from './components';

// HOC version
const CustomHocButtonBase: FC<InteractionTrackingProps> = ({
  children,
  interactionContent,
  trackInteraction,
}) => {
  const onClick = () => {
    if (interactionContent) {
      trackInteraction({
        interaction: InteractionType.Click,
        interactionContent,
      });
    }
  };

  return (
    <Layout margin={0.5}>
      <Button onClick={onClick} variant={ButtonType.Primary}>
        {children}
      </Button>
    </Layout>
  );
};

const CustomHocButton = withInteractionTracking(CustomHocButtonBase);

// Hook version
// Alter the `InteractionDataPayload` in whatever way best suits your
// component's use-case to create the right external API
interface CustomHookButtonProps
  extends Partial<Pick<InteractionDataPayload, 'interactionContent'>> {}

const CustomHookButton: FC<CustomHookButtonProps> = ({
  children,
  interactionContent,
}) => {
  const trackInteraction = useInteractionTracking();

  function onClick() {
    if (interactionContent) {
      trackInteraction({
        interaction: InteractionType.Click,
        interactionContent,
      });
    }
  }

  return (
    <Layout margin={0.5}>
      <Button onClick={onClick} variant={ButtonType.Primary}>
        {children}
      </Button>
    </Layout>
  );
};

// example
export const CustomInteractionExample: FC = () => (
  <ExtendInteractionMedium value="single-comp">
    <ScExampleSection>
      <ScText as="h3">
        Single component wrapped with withInteractionTracking HOC
      </ScText>
      <ScText>
        {`There's an ExtendInteractionMedium component surrounding this with value "single-comp". Below is a custom button that will get interaction content set as "custom-buttom" via manual onClick handling.`}
      </ScText>
      <CustomHocButton interactionContent="i-use-a-HOC">
        Click me to see my properties (HOC)
      </CustomHocButton>
      <CustomHookButton interactionContent="i-use-hooks">
        Click me to see my properties (hook)
      </CustomHookButton>
    </ScExampleSection>
  </ExtendInteractionMedium>
);
