import type { FC } from 'react';
import { useCallback, useMemo, useState } from 'react';
import ReactDOM from 'react-dom';
import styled from 'styled-components';
import type { ReportEvent } from 'tachyon-event-reporter';
import { EventReporterRoot, useReportEvent } from 'tachyon-event-reporter';
import type {
  EventTrackerRootProps,
  ProcessedInteractionEventData,
} from 'tachyon-event-tracker';
import { EventTrackerRoot } from 'tachyon-event-tracker';
import { CoreUiSsrRoot } from 'tachyon-more-ui';
import type { TwitchEvent } from 'tachyon-type-library';
import { ConnectionType, EffectiveConnectionType } from 'tachyon-type-library';
import { CustomInteractionExample } from './CustomInteractionExample';
import { LastEventData } from './LastEventData';
import { ListInteractionExample } from './ListInteractionExample';
import { ScCenteredText } from './components';

const ScContent = styled.div`
  display: flex;
`;

const ScMain = styled.main`
  max-height: 85vh;
  overflow-y: scroll;
  padding: 2rem;
`;

const ScAside = styled.aside`
  background: papayawhip;
  color: maroon;
  min-width: 250px;
  padding: 2rem;
`;

type DemoEventTrackerRootProps = {
  echoEventData: (data: any) => void;
};

const DemoEventTrackerRoot: FC<DemoEventTrackerRootProps> = ({
  children,
  echoEventData,
}) => {
  const reportEvent = useReportEvent();

  const eventHandler: ReportEvent = useCallback(
    (e) => {
      echoEventData(e);
      reportEvent(e);
    },
    [echoEventData, reportEvent],
  );

  const staticProperties: EventTrackerRootProps['staticProperties'] = useMemo(
    () => ({
      appVersion: 'demo',
      clientApp: 'tachyon-interaction-tracking-demo',
      deviceID: 'device id',
      language: 'en',
      netInfo: {
        bandwidth: 10,
        bandwidth_max: 100,
        effective_mobile_connection_type:
          EffectiveConnectionType.SecondGeneration,
        mobile_connection_type: ConnectionType.Bluetooth,
        round_trip_time: 75,
      },
      platform: 'tachyon',
      sessionID: 'session id',
    }),
    [],
  );

  return (
    <EventTrackerRoot
      children={children}
      interactionMedium="example"
      location="interaction-tracking-app"
      onEvent={eventHandler}
      staticProperties={staticProperties}
    />
  );
};

const InteractionTrackingExample: FC = () => {
  const [eventData, setEventData] = useState<TwitchEvent | null>(null);

  // use something like this for actual reporting:
  // <EventReporterRoot spade={{ spadeUrl: 'some url' }}>
  return (
    <CoreUiSsrRoot appRootElementId="root">
      <EventReporterRoot>
        <DemoEventTrackerRoot echoEventData={setEventData}>
          <ScCenteredText as="h1">Interaction Tracking Example</ScCenteredText>
          <ScContent>
            <ScMain>
              <h2>Possible implementation patterns:</h2>
              <CustomInteractionExample />
              <ListInteractionExample />
            </ScMain>
            <ScAside>
              <h4>Last Event Data</h4>
              <LastEventData
                eventData={eventData as ProcessedInteractionEventData}
              />
            </ScAside>
          </ScContent>
        </DemoEventTrackerRoot>
      </EventReporterRoot>
    </CoreUiSsrRoot>
  );
};

ReactDOM.render(
  <InteractionTrackingExample />,
  document.getElementById('root'),
);
