import type { FC } from 'react';
import ReactDOM from 'react-dom';
import styled from 'styled-components';
import {
  BandsInTownSvg,
  CoreUiSsrRoot,
  GlowErrorCarSvg,
  GlowGearSvg,
  GlowGeoBlockedSvg,
  GlowJoinTheConversation,
  GlowMatureGateSvg,
  GlowReportChannel,
  GlowSettingsHelp,
  GlowSettingsTermsOfService,
  GlowSubOnlySvg,
  GlowWatchAgainSvg,
  IOSAddToHomescreenIcon,
  IOSShareIcon,
  MixerSvg,
  QRUrl,
  ShadowedSVG,
  StreamType,
  StreamTypeIndicator,
  VkSvg,
  squareSize,
} from 'tachyon-more-ui';
import { Enum, useStorage } from 'tachyon-utils';
import {
  AlignItems,
  Background,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FontSize,
  JustifyContent,
  Layout,
  SVGAsset,
  ThemeToggle,
} from 'twitch-core-ui';

type ComponentDemo = {
  examples: JSX.Element[];
  name: string;
};

const ScExampleContainer = styled.div`
  min-height: 100vh;
  width: 100vw;
`;

const ScShadowedSVGFillContainer = styled.div`
  ${squareSize({ sizePx: 60 })}
`;

const ScGlowContainer = styled.div`
  align-items: center;
  display: flex;
  justify-content: center;
  ${squareSize({ sizePx: 100 })}
`;

const svgExamples: FC[] = [
  BandsInTownSvg,
  GlowErrorCarSvg,
  GlowGearSvg,
  GlowGeoBlockedSvg,
  GlowJoinTheConversation,
  GlowMatureGateSvg,
  GlowReportChannel,
  GlowSettingsHelp,
  GlowSettingsTermsOfService,
  GlowSubOnlySvg,
  GlowWatchAgainSvg,
  IOSAddToHomescreenIcon,
  IOSShareIcon,
  MixerSvg,
  VkSvg,
];

export const Example: FC = () => {
  const [theme, setTheme] = useStorage<'dark' | 'light'>(
    'more-ui-demo-theme',
    'light',
  );

  const components: ComponentDemo[] = [
    {
      examples: Enum.values(StreamType).map((streamType) => (
        <StreamTypeIndicator
          isOverlay={false}
          isPulsing
          key={streamType}
          streamType={streamType}
        />
      )),
      name: 'StreamTypeIndicator',
    },
    {
      examples: [
        SVGAsset.NavSearch,
        SVGAsset.Emoticons,
        SVGAsset.ViewerList,
      ].map((svg) => (
        <ShadowedSVG asset={svg} heightRem={4} key={svg} widthRem={4} />
      )),
      name: 'ShadowedSVG (sized)',
    },
    {
      examples: [SVGAsset.Discover, SVGAsset.Cake, SVGAsset.Moon].map((svg) => (
        <ScShadowedSVGFillContainer key={svg}>
          <ShadowedSVG asset={svg} fill />
        </ScShadowedSVGFillContainer>
      )),
      name: 'ShadowedSVG (fill)',
    },
    {
      examples: [
        <QRUrl
          content="content"
          key="QRUrl"
          medium="medium"
          sizeRem={30}
          url="https://example.com/"
        />,
      ],
      name: 'QRUrl',
    },
    {
      examples: svgExamples.sort().map((Svg) => {
        const name = Svg.displayName;
        return (
          <Layout
            alignItems={AlignItems.Center}
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            key={name}
            padding={{ right: 5 }}
          >
            <ScGlowContainer>
              <Svg />
            </ScGlowContainer>
            <CoreText>{name}</CoreText>
          </Layout>
        );
      }),
      name: 'Custom SVGs',
    },
  ];

  return (
    <CoreUiSsrRoot appRootElementId="root" theme={theme}>
      <ScExampleContainer>
        <Layout
          background={Background.Base}
          color={Color.Base}
          fullHeight
          fullWidth
          padding={5}
        >
          <Layout
            borderBottom
            display={Display.Flex}
            flexDirection={FlexDirection.Row}
            fullWidth
            justifyContent={JustifyContent.Between}
            margin={{ bottom: 2 }}
            padding={{ bottom: 1 }}
          >
            <CoreText fontSize={FontSize.Size3}>More UI components</CoreText>
            <ThemeToggle
              checked={theme === 'dark'}
              onChange={(e) =>
                setTheme(e.currentTarget.checked ? 'dark' : 'light')
              }
            />
          </Layout>

          {components.map(({ examples, name }) => (
            <Layout
              borderBottom
              key={name}
              margin={{ bottom: 1 }}
              padding={{ bottom: 2 }}
            >
              <CoreText fontSize={FontSize.Size4}>{`<${name} />`}</CoreText>
              <Layout
                display={Display.Flex}
                flexDirection={FlexDirection.Row}
                flexWrap={FlexWrap.Wrap}
                justifyContent={JustifyContent.Start}
                padding={{ top: 1 }}
              >
                {examples}
              </Layout>
            </Layout>
          ))}
        </Layout>
      </ScExampleContainer>
    </CoreUiSsrRoot>
  );
};

ReactDOM.render(<Example />, document.getElementById('root'));
