import type { FC } from 'react';
import ReactDOM from 'react-dom';
import styled from 'styled-components';
import { CoreUiSsrRoot } from 'tachyon-more-ui';
import type { Notification } from 'tachyon-notification';
import {
  NotificationContextRoot,
  useNotificationConsumer,
  useNotificationPublisher,
} from 'tachyon-notification';
import type { AlertBannerProps } from 'twitch-core-ui';
import {
  AlertBanner,
  AlertBannerType,
  Button,
  Display,
  FlexDirection,
  Layout,
  Position,
} from 'twitch-core-ui';

const bannerType: Record<Notification['type'], AlertBannerType> = {
  error: AlertBannerType.Error,
  info: AlertBannerType.Info,
  success: AlertBannerType.Success,
  warning: AlertBannerType.Warning,
};

type ToastParams = Pick<AlertBannerProps, 'message' | 'status'> & {
  dismissable: boolean;
};

function getNotificationContent(
  notification: Notification<ToastParams>,
): ToastParams {
  return notification.meta;
}

const ScToastGutter = styled(Layout)`
  width: 50rem;
  .tw-alert-banner {
    height: 4rem;
  }
`;

const Toaster: FC = () => {
  const notifications = useNotificationConsumer<ToastParams>({
    limit: 5,
    order: 'lifo',
  });

  return (
    <ScToastGutter attachRight fullHeight position={Position.Absolute}>
      {notifications.map((notification) => {
        const { dismissable, message, status } =
          getNotificationContent(notification);
        return (
          <Layout key={notification.id} margin={{ x: 2, y: 2 }}>
            <AlertBanner
              // TODO CORE-UI-TYPES: Update core-ui types to accept undefined
              // https://jira.xarth.tv/browse/EMP-4748
              closeButton={
                (dismissable
                  ? {
                      'aria-label': 'Close Banner',
                      onClick: notification.dismiss,
                    }
                  : undefined) as NonNullable<AlertBannerProps['closeButton']>
              }
              message={message}
              // TODO CORE-UI-TYPES: Update core-ui types to accept undefined
              // https://jira.xarth.tv/browse/EMP-4748
              status={status as string}
              type={bannerType[notification.type]}
            />
          </Layout>
        );
      })}
    </ScToastGutter>
  );
};

const Controls: FC = () => {
  const { publish } = useNotificationPublisher<ToastParams>();

  const examples = [
    {
      onClick: () =>
        publish({
          meta: {
            dismissable: true,
            message: 'I am a warning',
            status: 'Warning',
          },
          type: 'warning',
        }),
      title: 'Warning message',
    },
    {
      onClick: () =>
        publish({
          autoCloseMs: 5000,
          meta: {
            dismissable: false,
            message: 'I will auto dismiss in 5s',
            status: 'Auto Dismiss',
          },
          type: 'info',
        }),
      title: 'Auto dismiss after 5s',
    },
  ];

  return (
    <Layout
      borderBottom
      display={Display.Flex}
      flexDirection={FlexDirection.Row}
      fullWidth
      padding={1}
    >
      {examples.map(({ onClick, title }) => (
        <Layout key={title} margin={{ x: 1 }}>
          <Button children={title} onClick={onClick} />
        </Layout>
      ))}
    </Layout>
  );
};

export const Example: FC = () => (
  <CoreUiSsrRoot appRootElementId="root">
    <Layout fullHeight fullWidth>
      <NotificationContextRoot>
        <Controls />
        <Toaster />
      </NotificationContextRoot>
    </Layout>
  </CoreUiSsrRoot>
);

ReactDOM.render(<Example />, document.getElementById('root'));
