import { VideoType, browserSupportsHLS } from 'pulsar';
import type { FC } from 'react';
import { Suspense } from 'react';
import { RelayEnvironmentProvider } from 'react-relay/hooks';
import { CoreUiSsrRoot } from 'tachyon-more-ui';
import { configureTachyonRelay, initEnvironment } from 'tachyon-relay';
import { useConst, useStorage } from 'tachyon-utils';
import {
  Display,
  FlexDirection,
  Layout,
  LoadingSpinner,
  SegmentedButton,
  SegmentedButtonOption,
  SpinnerSize,
} from 'twitch-core-ui';
import { CLIENT_ID } from '../../config';
import { ClipPage } from '../pages/ClipPage';
import { StreamPage } from '../pages/StreamPage';
import { VideoPage } from '../pages/VideoPage';
import type { CommonPageProps } from '../types';

configureTachyonRelay({ clientId: CLIENT_ID });

export const Router: FC = () => {
  const relayEnv = useConst(initEnvironment);
  const appInitTimeMS = useConst(Date.now);

  const [playerBackend, setPlayerBackend] = useStorage<
    CommonPageProps['playerBackend']
  >('pulsar-example-last-player-backend', 'mediaplayer');

  const [selectedVideoType, setVideoType] = useStorage(
    'pulsar-example-last-video-type',
    VideoType.Stream,
  );

  const PageToRender = (() => {
    switch (selectedVideoType) {
      case VideoType.Stream:
        return StreamPage;
      case VideoType.Clip:
        return ClipPage;
      case VideoType.Video:
        return VideoPage;
    }
  })();

  return (
    <RelayEnvironmentProvider environment={relayEnv}>
      <CoreUiSsrRoot appRootElementId="root">
        <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
          <Layout display={Display.Flex} padding={1}>
            <SegmentedButton>
              <SegmentedButtonOption
                checked={selectedVideoType === VideoType.Stream}
                label="Stream"
                name="stream"
                onChange={() => {
                  setVideoType(VideoType.Stream);
                }}
              />
              <SegmentedButtonOption
                checked={selectedVideoType === VideoType.Video}
                label="Video"
                name="vod"
                onChange={() => {
                  setVideoType(VideoType.Video);
                }}
              />
              <SegmentedButtonOption
                checked={selectedVideoType === VideoType.Clip}
                label="Clip"
                name="clip"
                onChange={() => {
                  setVideoType(VideoType.Clip);
                }}
              />
            </SegmentedButton>
            <Layout padding={{ left: 1 }}>
              <SegmentedButton>
                <SegmentedButtonOption
                  checked={playerBackend === 'mediaplayer'}
                  label="Player-Core"
                  name="mediaplayer"
                  onChange={() => {
                    setPlayerBackend('mediaplayer');
                  }}
                />
                {browserSupportsHLS() && (
                  <SegmentedButtonOption
                    checked={playerBackend === 'native'}
                    label="Native HLS"
                    name="native"
                    onChange={() => {
                      setPlayerBackend('native');
                    }}
                  />
                )}
                {/* HLS does not support Clips due to not supporting mp4 sources */}
                {selectedVideoType !== VideoType.Clip && (
                  <SegmentedButtonOption
                    checked={playerBackend === 'hlsjs'}
                    label="Hls.js"
                    name="hlsjs"
                    onChange={() => {
                      setPlayerBackend('hlsjs');
                    }}
                  />
                )}
              </SegmentedButton>
            </Layout>
          </Layout>
          <Suspense
            fallback={<LoadingSpinner fillContent size={SpinnerSize.Large} />}
          >
            <PageToRender
              appInitTimeMS={appInitTimeMS}
              playerBackend={playerBackend}
            />
          </Suspense>
        </Layout>
      </CoreUiSsrRoot>
    </RelayEnvironmentProvider>
  );
};
