import { useMuted, useVolume } from 'pulsar';
import type { ChangeEvent, FC } from 'react';
import { useRef } from 'react';
import {
  AlignItems,
  ButtonIcon,
  Display,
  Layout,
  Range,
  SVGAsset,
} from 'twitch-core-ui';

function volumeFromRange(input: HTMLInputElement): number {
  return Number.parseFloat(input.value);
}

export const VolumeControl: FC = () => {
  const { muted, setMuted } = useMuted();
  const { setVolume, volume } = useVolume();
  const rangeRef = useRef<HTMLInputElement | null>(null);

  const preMuteVolumeRef = useRef<number | null>(null);

  function toggleMute(): void {
    if (!rangeRef.current) {
      return;
    }

    const willMute = !muted;
    if (willMute) {
      preMuteVolumeRef.current = volumeFromRange(rangeRef.current);
      rangeRef.current.value = '0.0';
    } else {
      rangeRef.current.value = (preMuteVolumeRef.current ?? 0.1).toString();

      // Clear last value
      preMuteVolumeRef.current = null;
    }

    setMuted(!muted);
  }

  function volumeChangeHandler(e: ChangeEvent<HTMLInputElement>): void {
    const newVolume = volumeFromRange(e.target);

    // Mute when almost 0 since it's hard to get the range value to exactly 0
    if (!muted && newVolume <= 0.01) {
      setMuted(true);
    }

    if (muted && newVolume >= 0.01) {
      // clear this value if it was set since the user manually un-muted using the slider
      preMuteVolumeRef.current = null;
      setMuted(false);
    }

    setVolume(newVolume);
  }

  return (
    <Layout
      alignItems={AlignItems.Center}
      display={Display.Flex}
      padding={{ left: 0.5 }}
    >
      <ButtonIcon
        aria-label={muted ? 'unmute' : 'mute'}
        icon={muted ? SVGAsset.VolumeMuted : SVGAsset.VolumeMax}
        onClick={toggleMute}
      />
      <Range
        defaultValue={volume.toString()}
        max={1.0}
        min={0}
        onChange={volumeChangeHandler}
        refDelegate={rangeRef}
        step={0.01}
      />
    </Layout>
  );
};
