import {
  ClosedCaptionsRoot,
  PlayerControllerRoot,
  usePlayerError,
} from 'pulsar';
import type { FC } from 'react';
import { useEffect, useState } from 'react';
import styled from 'styled-components';
import { Display, JustifyContent, Layout } from 'twitch-core-ui';
import { ClosedCaptions } from './ClosedCaptions';
import { CustomPlayerControls } from './CustomPlayerControls';
import { PlayerOverlay } from './PlayerOverlay';
import { PlayerConsumerRoot } from './PulsarConsumerRoot';
import { EventTable } from './TestSuite';
import { useEventMarker } from './TestSuite/useEventMarker';

export * from './backends';

type PulsarConsumerProps = {
  appInitTimeMS: number;
  captionContentKey: string;
  displayVideoTime: boolean;
};

const ScPlayerContainer = styled.div`
  display: flex;
  flex-direction: column;
  margin-right: 0.5rem;
  position: relative;
  width: 70vw;
`;

export const PulsarConsumerBase: FC<PulsarConsumerProps> = ({
  appInitTimeMS,
  captionContentKey,
  children,
  displayVideoTime,
}) => {
  const { clearEvents, eventMap, markEvent } = useEventMarker(appInitTimeMS);
  const [showControls, setShowControls] = useState(true);
  const [liveLowLatencyEnabled, setLiveLowLatencyEnabled] = useState(true);
  const playerError = usePlayerError();

  function toggleControls(): void {
    setShowControls(!showControls);
  }

  function toggleLiveLowLatency(): void {
    setLiveLowLatencyEnabled((s) => !s);
  }

  useEffect(() => {
    if (playerError) {
      window.alert(`Player Error: ${JSON.stringify(playerError)}`);
    }
  }, [playerError]);

  return (
    <PlayerConsumerRoot
      liveLowLatencyEnabled={liveLowLatencyEnabled}
      markEvent={markEvent}
      showControls={showControls}
    >
      <PlayerControllerRoot>
        <Layout
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          margin={1}
        >
          <ClosedCaptionsRoot
            contentKey={captionContentKey}
            defaultEnabled={false}
          >
            <div>
              <ScPlayerContainer>
                <PlayerOverlay />
                {children}
                <ClosedCaptions />
              </ScPlayerContainer>
              <CustomPlayerControls
                displayVideoTime={displayVideoTime}
                onControlsToggle={toggleControls}
                onLowLatencyToggle={toggleLiveLowLatency}
              />
            </div>
          </ClosedCaptionsRoot>
          <EventTable eventMap={eventMap} onEventsCleared={clearEvents} />
        </Layout>
      </PlayerControllerRoot>
    </PlayerConsumerRoot>
  );
};

export const PulsarConsumer: FC<PulsarConsumerProps> = (props) => (
  <PlayerControllerRoot>
    <PulsarConsumerBase {...props} />
  </PlayerControllerRoot>
);
