// We use babel plugin macros here to work with CRA. Normally you would use
// `import { graphql } from 'react-relay';` with a proper babel config.
// @ts-expect-error: no type-defs available
import graphql from 'babel-plugin-relay/macro';
import { PlayerType, createClipManifestUrl, useClipAnalytics } from 'pulsar';
import type { PulsarProps } from 'pulsar';
import type { FC } from 'react';
import { useFragment } from 'react-relay/hooks';
import { useStaticEnvironment } from 'tachyon-environment';
import { exhaustedCase } from 'tachyon-utils';
import {
  PulsarCoreBackend,
  PulsarHlsjsBackend,
  PulsarNativeBackend,
} from '../../common/PulsarConsumer';
import { usePlayerConsumer } from '../../common/PulsarConsumer/PulsarConsumerRoot';
import type { CommonPageProps } from '../../types';
import type { ClipPlayer_clip$key } from './__generated__/ClipPlayer_clip.graphql';

const clipPlayerFragment = graphql`
  fragment ClipPlayer_clip on Clip {
    id
    slug
    game {
      name
    }
    broadcaster {
      id
      login
      roles {
        isPartner
      }
      self {
        subscriptionBenefit {
          id
        }
      }
    }
    playbackAccessToken(
      params: {
        playerType: "pulsar"
        playerBackend: "mediaplayer"
        platform: "web"
      }
    ) {
      signature
      value
    }
    videoQualities {
      quality
      sourceURL
    }
  }
`;

export type ClipPlayerProps = Partial<PulsarProps> & {
  clip: ClipPlayer_clip$key;
  playerBackend: CommonPageProps['playerBackend'];
};

export const ClipPlayer: FC<ClipPlayerProps> = ({
  clip: clipRef,
  playerBackend,
  ...pulsarProps
}) => {
  const clipData = useFragment(clipPlayerFragment, clipRef);

  const { markEvent } = usePlayerConsumer();
  const {
    client: { sessionID },
  } = useStaticEnvironment();

  useClipAnalytics({
    clip: clipData,
    currentUser: null,
    onTrackingEvent: markEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
  });

  const manifestUrl = createClipManifestUrl(clipData);
  if (!manifestUrl) {
    throw new Error('Unable to create Clip manifest URL');
  }

  switch (playerBackend) {
    case 'mediaplayer':
      return <PulsarCoreBackend src={manifestUrl} {...pulsarProps} />;
    case 'native':
      return <PulsarNativeBackend src={manifestUrl} {...pulsarProps} />;
    case 'hlsjs':
      return <PulsarHlsjsBackend src={manifestUrl} {...pulsarProps} />;
    default:
      return exhaustedCase(playerBackend, null);
  }
};
