// We use babel plugin macros here to work with CRA. Normally you would use
// `import { graphql } from 'react-relay';` with a proper babel config.
// @ts-expect-error: no type-defs available
import graphql from 'babel-plugin-relay/macro';
import type { FC } from 'react';
import { useState } from 'react';
import { useLazyLoadQuery } from 'react-relay/hooks';
import { AlignItems, CheckBox, Display, Layout } from 'twitch-core-ui';
import { PulsarConsumer } from '../../common/PulsarConsumer';
import type { CommonPageProps } from '../../types';
import { ClipPlayer } from './ClipPlayer';
import { ClipSetter } from './ClipSetter';
import type { ClipPage_Query } from './__generated__/ClipPage_Query.graphql';

// Consistent fragment for video
// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ClipPage_clip on Clip {
    id
    slug
  }
`;

const clipPageQuery = graphql`
  query ClipPage_Query($slug: ID!, $noClip: Boolean!) {
    topClip: user(login: "timthetatman") @include(if: $noClip) {
      clips(first: 1) {
        edges {
          node {
            ...ClipPage_clip @relay(mask: false)
            ...ClipPlayer_clip
          }
        }
      }
    }
    clip(slug: $slug) @skip(if: $noClip) {
      ...ClipPage_clip @relay(mask: false)
      ...ClipPlayer_clip
    }
  }
`;

type ClipPageProps = CommonPageProps;

export const ClipPage: FC<ClipPageProps> = ({
  appInitTimeMS,
  playerBackend,
}) => {
  const [selectedClipSlug, setSelectedClipSlug] = useState<string | null>(null);
  const [loop, setLoop] = useState(true);
  const [autoPlay, setAutoPlay] = useState(true);

  const data = useLazyLoadQuery<ClipPage_Query>(clipPageQuery, {
    noClip: selectedClipSlug === null,
    slug: selectedClipSlug ?? '',
  });

  const clipData = data?.clip ?? data?.topClip?.clips?.edges?.[0]?.node ?? null;
  if (!clipData) {
    throw new Error('No Clip Data 🥺');
  }

  function onLoopToggle() {
    setLoop(!loop);
  }

  return (
    <>
      <Layout alignItems={AlignItems.Center} display={Display.Flex} padding={2}>
        <ClipSetter
          defaultClipSlug={clipData.slug}
          onContentChange={setSelectedClipSlug}
        />
        <Layout padding={{ x: 1 }}>
          <CheckBox defaultChecked label="Loop" onChange={onLoopToggle} />
        </Layout>
        <CheckBox
          checked={autoPlay}
          label="Autoplay"
          onChange={(event) => {
            setAutoPlay(event.currentTarget.checked);
          }}
        />
      </Layout>
      <PulsarConsumer
        appInitTimeMS={appInitTimeMS}
        captionContentKey={`clip-${clipData.id}`}
        displayVideoTime
      >
        <ClipPlayer
          autoPlay={autoPlay}
          clip={clipData}
          loop={loop}
          playerBackend={playerBackend}
        />
      </PulsarConsumer>
    </>
  );
};
